# (c) 2025 Ricci Adams
# MIT License (or) 1-clause BSD License

# Don't evaluate type annotations at runtime
from __future__ import annotations

import sublime
import sublime_plugin
import os
import shutil


# Bump this after adding a new icons
Version = 2


FileSuffixToScopeMap = {
    "c":          "source.c",
    "c++":        "source.c++",
    "clojure":    "source.clojure",
    "csharp":     "source.cs",
    "css":        "source.css",
    "go":         "source.go",
    "html":       "text.html.basic, text.html.plain",
    "image":      "binary.image",
    "java":       "source.java",
    "js":         "source.js",
    "json":       "source.json",
    "lisp":       "source.autolisp, source.lisp",
    "lua":        "source.lua",
    "nyx":        "source.js.nyx",
    "objc":       "source.objc",
    "objc++":     "source.objc++",
    "perl":       "source.perl",
    "php":        "embedding.php, source.php",
    "python":     "source.python, text.plain.python",
    "ruby":       "source.ruby, source.shell.ruby, text.plain.rbs, text.plain.ruby",
    "rust":       "source.rust",
    "swift":      "source.swift",
    "typescript": "source.js.typescript, source.ts",
}


CLetterFileSuffixToInfoMap = {
    "h":   ( "h",   "source.c++.header" ),
    "c":   ( "c",   "source.c" ),
    "c++": ( "cpp", "source.c++" ),
    "m":   ( "m",   "source.objc" ),
    "mm":  ( "mm",  "source.objc++" )
}


AboutTxt = """
This folder was created by the Timeless theme to enable
additional custom file icons.

Due to Sublime Text's implementation of file icons, the
contents of this folder may break icons in other themes.

For more information, visit:
https://github.com/iccir/Timeless-Theme/issues/6
"""


TmPreferencesTemplate = """<?xml version="1.0" encoding="UTF-8"?>
<plist version="1.0">
    <dict>
        <key>scope</key>
        <string>%s</string>
        <key>settings</key>
        <dict>
            <key>icon</key>
            <string>%s</string>
        </dict>
    </dict>
</plist>"""


CPlusPlusHeaderSublimeSyntax = """%YAML 1.2
---
name: C++ Header
scope: source.c++.header
hidden: true

file_extensions:
  - h
  - hh
  - hpp
  - hxx
  - h++

extends: Packages/C++/C++.sublime-syntax
"""


CPlusPlusHeaderSublimeSettings = """{
    "extensions": [ "h" ]
}"""


CPlusPlusSublimeSettings = """{
    /*
        This file was created by the Timeless theme
        to enable the "H" icon for `.h` files. 
    */
    "extensions": [ "__timeless_theme_icon_support__" ]
}"""
        

PreferencesSettings = None


def get_timeless_icon_support_path() -> str:
    return os.path.join(sublime.cache_path(), "Timeless Icon Support")


def maybe_load_resource(name: str, default_value = None) -> Optional[str]:
    try:
        result = sublime.load_resource(name)
    except:
        result = default_value
    
    return result


def patch_cpp_extensions(enable: boolean):
    cpp_settings_name = "C++.sublime-settings"
    user_cpp_settings_path = os.path.join(sublime.packages_path(), "User", "C++.sublime-settings")
    magic_key = "__timeless_theme_icon_support__"

    if enable:
        if not os.path.exists(user_cpp_settings_path):
            with open(user_cpp_settings_path, "w") as f:
                f.write(CPlusPlusSublimeSettings)
        else:
            cpp_settings = sublime.load_settings(cpp_settings_name)
            cpp_settings.set("extensions", [ magic_key ])
            sublime.save_settings(cpp_settings_name)

    else:
        cpp_settings = sublime.load_settings(cpp_settings_name)
        if magic_key in cpp_settings.get("extensions", [ ]):
            del cpp_settings["extensions"]
            sublime.save_settings(cpp_settings_name)
            
            # Directly load the JSONC file and see if it is empty.
            # If so, remove it to properly clean up after ourselves.
            raw_dict = sublime.decode_value(maybe_load_resource("Packages/User/C++.sublime-settings", ""))
            
            if raw_dict == { }:
                os.remove(user_cpp_settings_path)


def unassign_support_syntax() -> None:
    for w in sublime.windows():
        for v in w.views(include_transient=True):
            syntax = v.syntax()
            if not syntax: continue
            if not syntax.path: continue
            
            if "Timeless Icon Support" in syntax.path:
                v.assign_syntax("scope:text.plain")


def get_version_string(c_letters: bool) -> str:
    if c_letters:
        return "{}+c_letters".format(Version)
    else:
        return "{}".format(Version)


def is_version_installed(version_path: str, version_string: str) -> bool:
    version_installed = None

    try:
        with open(version_path, "r") as f:
            version_installed = f.read().strip()
    except:
        pass

    return version_installed == version_string
    

def install_icon_support(c_letters: bool = False) -> None:
    base_path = get_timeless_icon_support_path()
    files_path = os.path.join(base_path, "files")

    version_path = os.path.join(files_path, "version.txt")
    version_string = get_version_string(c_letters)

    if is_version_installed(version_path, version_string):
        return

    remove_icon_support()

    os.mkdir(base_path)
    os.mkdir(files_path)
    
    with open(os.path.join(base_path, "about.txt"), "w") as f:
        f.write(AboutTxt)
        
    with open(version_path, "w") as f:
        f.write(version_string)

    files_to_write = { }
    
    def write_file(path, contents):
        with open(os.path.join(files_path, path), "w") as f:
            f.write(contents)

    for key, value in FileSuffixToScopeMap.items():
        contents = TmPreferencesTemplate % (value, f"file_type_{key}")
        files_to_write[f"icon_{key}.tmPreferences"] = contents
        
    if c_letters:
        for key, value in CLetterFileSuffixToInfoMap.items():
            letter, scope = value
            contents = TmPreferencesTemplate % (scope, f"timeless_letter_{letter}")
            files_to_write[f"icon_{key}.tmPreferences"] = contents

        write_file("C++ Header.sublime-syntax",   CPlusPlusHeaderSublimeSyntax)
        write_file("C++ Header.sublime-settings", CPlusPlusHeaderSublimeSettings)
        
        patch_cpp_extensions(True)

    for key, value in files_to_write.items():
        write_file(key, value)


def remove_icon_support() -> None:
    unassign_support_syntax()

    try:
        shutil.rmtree(get_timeless_icon_support_path(), ignore_errors=True)
    except:
        pass

    patch_cpp_extensions(False)


cached_install_support_files = None
cached_install_c_letters     = None

def handle_settings_change():
    global cached_install_support_files, cached_install_c_letters

    install_support_files = PreferencesSettings.get("theme.timeless.icons.install_support_files", False)
    install_c_letters     = PreferencesSettings.get("theme.timeless.icons.install_c_letters",     False)
    
    if (
        install_support_files == cached_install_support_files and
        install_c_letters     == cached_install_c_letters
    ):
        return
    
    if install_support_files:
        install_icon_support(c_letters = install_c_letters)
    else:
        remove_icon_support()
        
    cached_install_support_files = install_support_files
    cached_install_c_letters     = install_c_letters


def plugin_loaded() -> None:
    global PreferencesSettings

    PreferencesSettings = sublime.load_settings("Preferences.sublime-settings")
    PreferencesSettings.add_on_change("Timeless.IconSupport", handle_settings_change)

    handle_settings_change()


def plugin_unloaded() -> None:
    if PreferencesSettings:
        PreferencesSettings.clear_on_change("Timeless.IconSupport")

    try:
        from package_control import events
        if events.remove("Theme - Timeless"):
            remove_icon_support()
    except:
        pass

