import sublime
import sublime_plugin
import subprocess
import os
import threading
import fnmatch

SETTINGS_FILE = "SimpleSync.sublime-settings"
LOG_PREFIX = "SimpleSync: "

# print messages functions
def show_status(view, msg):
    if view:
        view.set_status(LOG_PREFIX, msg)
        # Clear after 5 seconds
        sublime.set_timeout(lambda: view.erase_status(LOG_PREFIX), 5000)
    else:
        sublime.status_message(msg)

def show_error_msg(view, msg):
    print(LOG_PREFIX + msg)
    settings = sublime.load_settings(SETTINGS_FILE)

    # get what type of notification to show
    notification = settings.get("upload_error_notification", "")

    if not notification:
        return

    def _show():
        if notification == "popup":
            window = view.window()
            if window and window.active_view():
                window.active_view().show_popup(
                    LOG_PREFIX+"<b style='color:red'>File not synced</b><br><br>" + msg,
                    max_width=1000
                )
        elif notification == "dialog":
            sublime.message_dialog(LOG_PREFIX + "File not synced\n\n" + msg)
        else:
            print(LOG_PREFIX + "invalid upload_error_notification setting: " + notification)

    sublime.set_timeout(_show, 0)

# functions to check criteria to sync
def is_in_sync_folders(file_path):

    settings = sublime.load_settings(SETTINGS_FILE)
    local_folder = settings.get("local_folder")
    subfolders = settings.get("subfolders_to_sync")
    
    if not local_folder:
        return False

    file_path = os.path.abspath(file_path)
    if subfolders:
        for subfolder in subfolders:
            # strip leading / if exist in subfolder otherwise os.path.join will consider it as absolute path
            subfolder = subfolder.lstrip(os.sep)
            folder = os.path.abspath(os.path.join(local_folder, subfolder))
            if file_path.startswith(folder + os.sep):
                return True
    else:
        folder = os.path.abspath(local_folder)
        if file_path.startswith(folder + os.sep):
            return True

    
    return False


def is_excluded(file_path):
    """
    Check if file_path matches any of the exclude patterns.
    Supports glob patterns like: *.log, *.tmp, __pycache__/*, .git/*
    """
    settings = sublime.load_settings(SETTINGS_FILE)
    patterns = settings.get("exclude_patterns", [])
    if not patterns:
        return False

    file_path = os.path.normpath(file_path)
    filename = os.path.basename(file_path)

    for pattern in patterns:
        if fnmatch.fnmatch(filename, pattern):
            return True
        # Match against full path (for patterns like __pycache__/*)
        if fnmatch.fnmatch(file_path, pattern):
            return True
        # Check if pattern appears anywhere in path
        if '*' not in pattern and pattern in file_path:
            return True

    return False

# run command with timeout
def run_command(cmd, timeout):
    proc = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)

    # Kill process if it exceeds timeout
    timer = threading.Timer(timeout, proc.kill)
    timer.start()

    stdout, stderr = proc.communicate()

    if timer:
        timer.cancel()

    if proc.returncode != 0:
        raise subprocess.CalledProcessError(proc.returncode, cmd, output=stdout + stderr)

    return stdout, stderr

# core function
def upload_file(file_path, view):
    settings = sublime.load_settings(SETTINGS_FILE)
    REMOTE_USER = settings.get("remote_user")
    REMOTE_HOST = settings.get("remote_host")
    REMOTE_FOLDER = settings.get("remote_folder")
    LOCAL_FOLDER = settings.get("local_folder")

    #optional configuration
    PASSWORD_FILE = settings.get("password_file","")
    KEY_FILE = settings.get("key_file","")
    TIMEOUT = settings.get("timeout",10)
    REMOTE_PORT = settings.get("remote_port","")

    # Validate required settings
    if not all([REMOTE_USER, REMOTE_HOST, REMOTE_FOLDER, LOCAL_FOLDER]):
        show_error_msg(view, "Missing required settings")
        return

    # set the relative paths against the remote and local base
    relative_path = os.path.relpath(file_path, LOCAL_FOLDER)
    # change \ to / for syncing from windows (local) to linux (remote)
    remote_path = REMOTE_FOLDER + "/" + relative_path.replace("\\", "/")

    """
    build scp command. Depends on configuration
    KEY_FILE: "scp -i /path/to/key"
    PASSWORD_FILE: "sshpass -f /path/to/password scp"
    default: "scp"
    """
    scp_command = ["scp"]
    if KEY_FILE:
        scp_command += ["-i", KEY_FILE]
    elif PASSWORD_FILE:
        if sublime.platform() == "windows":
            show_error_msg(view,"sshpass not supported in Windows")
            return
        else:
            scp_command = ["sshpass", "-f", PASSWORD_FILE] + scp_command

    # add port
    if REMOTE_PORT:
        scp_command += ["-P", REMOTE_PORT]

    # complete scp command i.e "[scp] /file/to/sync remote_user@remote_ip:/remote/file/to/sync"
    scp_command += [file_path, REMOTE_USER + "@" + REMOTE_HOST + ":" + remote_path]

    try:
        run_command(scp_command, TIMEOUT)


    except subprocess.CalledProcessError as e:
        msg = "failed to upload: " + file_path + " error: "
        if e.output: 
            msg += e.output.decode()
        else:
            msg += "check server reachability"
        show_error_msg(view, msg)


class SimpleSyncSaveListener(sublime_plugin.EventListener):

    def on_post_save(self, view):
        settings = sublime.load_settings(SETTINGS_FILE)

        file_path = view.file_name()
        if not file_path:
            return

        if not settings.get("enabled", True):
            return

        if not is_in_sync_folders(file_path):
            show_status(view, "skip " + file_path + " - File not in sync folders.")
            return

        if is_excluded(file_path):
            show_status(view, "skip " + file_path + " - File pattern is excluded")
            return

        # Run upload in background thread
        threading.Thread(
            target=upload_file,
            args=(file_path, view),
            daemon=True
        ).start()
