import shutil
import sys

from pathlib import Path
from typing import cast

import sublime

from .color_schemes import clear_color_schemes
from .color_schemes import clear_invalid_color_schemes
from .color_schemes import select_color_scheme

BOOTSTRAP_VERSION = "3.0.3"


def augment_default_markdown():
    """
    Augment ST's default Markdown.sublime-package with MDE's syntaxes.

    As of ST 4134 builtin Markdown syntax can be used as base syntax for various packages.
    Hence disabling default package may break at least HAML, Astro, Liquid and probably more.

    To avoid that from happening, MDE creates a Markdown.sublime-package to completely
    override default package instead. This step prevents MarkdownEditing from extending
    default Markdown syntax, but it's probably the most safest way to provide all its
    benefits (additional fenced code blocks) to all other 3rd-party packages.
    """
    from ..plugin import __version__

    installed_packages = Path(sublime.installed_packages_path())
    pkg_path = installed_packages / "Markdown.sublime-package"

    try:
        if pkg_path.exists():
            augmented_version = sublime.load_resource("Packages/Markdown/AUTOGENERATED")
            if augmented_version == __version__:
                return
    except FileNotFoundError:
        pass

    from textwrap import dedent
    from zipfile import ZipFile, ZIP_DEFLATED

    def load_res(syntax):
        return (
            sublime.load_resource(f"Packages/MarkdownEditing/syntaxes/{syntax}")
            .replace("\r\n", "\n")
            .replace("\r", "\n")
        )

    new_path = installed_packages / "Markdown.sublime-package-new"

    try:
        with ZipFile(new_path, "w", compression=ZIP_DEFLATED) as pkg:
            # copy unhidden Markdown syntax and assign default file extensions
            pkg.writestr(
                "Markdown.sublime-syntax",
                load_res("Markdown.sublime-syntax").replace(
                    "\nhidden: true\n",
                    dedent(
                        """

                        file_extensions:
                          - md
                          - mdown
                          - markdown
                          - markdn
                        """
                    ),
                ),
            )
            # copy unhidden MultiMarkdown syntax
            pkg.writestr(
                "MultiMarkdown.sublime-syntax",
                load_res("MultiMarkdown.sublime-syntax").replace("\nhidden: true\n", "\n"),
            )
            pkg.writestr(
                "README.md",
                dedent(
                    """
                    This package is auto-generated by MarkdownEditing
                    to augment ST's default Markdown syntax.
                    """
                ).lstrip(),
            )
            pkg.writestr("AUTOGENERATED", __version__)

    except OSError:
        new_path.unlink(True)
        return

    # replace package by renaming to workaround file locking on Windows
    old_path = installed_packages / "Markdown.sublime-package-old"
    if pkg_path.exists():
        old_path.unlink(True)
        pkg_path.rename(old_path)
    new_path.rename(pkg_path)
    sublime.set_timeout_async(lambda: old_path.unlink(True), 100)

    # remove Markdown.sublime-package from ignored packages
    settings = sublime.load_settings("Preferences.sublime-settings")
    ignored = cast(list, settings.get("ignored_packages", []))
    if ignored and "Markdown" in ignored:
        ignored.remove("Markdown")
        settings.set("ignored_packages", ignored)
        sublime.save_settings("Preferences.sublime-settings")


def restore_default_markdown():
    pkg_path = Path(sublime.installed_packages_path()) / "Markdown.sublime-package"
    pkg_path.unlink(True)


def on_after_install():
    augment_default_markdown()

    cache_path = Path(sublime.cache_path()) / "MarkdownEditing"
    bootstrapped = cache_path / "bootstrapped"

    # Check bootstrapped cookie.
    try:
        if bootstrapped.read_text() == BOOTSTRAP_VERSION:
            return
    except FileNotFoundError:
        pass

    # Clear previous syntax caches.
    shutil.rmtree(cache_path, ignore_errors=True)
    cache_path.mkdir(parents=True, exist_ok=True)

    def async_worker():
        clear_invalid_color_schemes()
        # Update bootstrap cookie.
        bootstrapped.write_text(BOOTSTRAP_VERSION)

        select_color_scheme()

    sublime.set_timeout_async(async_worker, 200)


def on_before_uninstall():
    restore_default_markdown()

    try:
        from package_control import events
    except Exception:
        pass
    else:
        if events.remove("MarkdownEditing"):
            clear_color_schemes()
