import os
import shutil
import sys

import sublime

from .color_schemes import clear_color_schemes, clear_invalid_color_schemes, select_color_scheme

BOOTSTRAP_VERSION = "3.0.3"

package_name = __package__.split(".")[0]


def augment_default_markdown():
    """
    Augment ST's default Markdown.sublime-package with MDE's syntaxes.

    As of ST 4134 builtin Markdown syntax can be used as base syntax for various packages.
    Hence disabling default package may break at least HAML, Astro, Liquid and probably more.

    To avoid that from happening, MDE creates a Markdown.sublime-package to completely
    override default package instead. This step prevents MarkdownEditing from extending
    default Markdown syntax, but it's probably the most safest way to provide all its
    benefits (additional fenced code blocks) to all other 3rd-party packages.
    """

    dst_path = os.path.join(sublime.installed_packages_path(), "Markdown.sublime-package")
    if os.path.isfile(dst_path):
        return

    from textwrap import dedent
    from zipfile import ZipFile, ZIP_DEFLATED

    def load_res(syntax):
        return (
            sublime.load_resource("/".join(("Packages", package_name, "syntaxes", syntax)))
            .replace("\r\n", "\n")
            .replace("\r", "\n")
        )

    tmp_path = os.path.join(sublime.installed_packages_path(), "Markdown.tmp")
    with ZipFile(tmp_path, "w", compression=ZIP_DEFLATED) as pkg:
        # copy unhidden Markdown syntax and assign default file extensions
        pkg.writestr(
            "Markdown.sublime-syntax",
            load_res("Markdown.sublime-syntax").replace(
                "\nhidden: true\n",
                dedent(
                    """

                    file_extensions:
                      - md
                      - mdown
                      - markdown
                      - markdn
                    """
                ),
            ),
        )
        # copy unhidden MultiMarkdown syntax
        pkg.writestr(
            "MultiMarkdown.sublime-syntax",
            load_res("MultiMarkdown.sublime-syntax").replace("\nhidden: true\n", "\n"),
        )
        pkg.writestr(
            "Shell (for Markdown).sublime-syntax",
            load_res("Shell (for Markdown).sublime-syntax"),
        )
        pkg.writestr(
            "README.md",
            dedent(
                """
                NOTE
                ====

                This file is auto generated by MarkdownEditing
                to augment ST's default Markdown syntax.
                """
            ).lstrip(),
        )
    os.rename(tmp_path, dst_path)

    prefs = sublime.load_settings("Preferences.sublime-settings")
    ignored = prefs.get("ignored_packages", [])
    if ignored and "Markdown" in ignored:
        ignored.remove("Markdown")
        prefs.set("ignored_packages", ignored)
        sublime.save_settings("Preferences.sublime-settings")


def restore_default_markdown():
    try:
        os.remove(os.path.join(sublime.installed_packages_path(), "Markdown.sublime-package"))
    except OSError:
        pass


def on_after_install():
    augment_default_markdown()

    cache_path = os.path.join(sublime.cache_path(), package_name)
    bootstrapped = os.path.join(cache_path, "bootstrapped")

    # Check bootstrapped cookie.
    try:
        if open(bootstrapped).read() == BOOTSTRAP_VERSION:
            return
    except FileNotFoundError:
        pass

    # Clear previous syntax caches.
    shutil.rmtree(cache_path, ignore_errors=True)
    os.makedirs(cache_path, exist_ok=True)

    def async_worker():
        clear_invalid_color_schemes()
        # Update bootstrap cookie.
        open(bootstrapped, "w").write(BOOTSTRAP_VERSION)

        select_color_scheme()

    sublime.set_timeout_async(async_worker, 200)


def on_before_uninstall():
    restore_default_markdown()

    if "package_control" in sys.modules:
        from package_control import events

        if events.remove(package_name):
            clear_color_schemes()
