r"""
hashlib module - A common interface to many hash functions.

new(name, data=b'', **kwargs) - returns a new hash object implementing the
                                given hash function; initializing the hash
                                using the given binary data.

Named constructor functions are also available, these are faster
than using new(name):

md5(), sha1(), sha224(), sha256(), sha384(), sha512(), blake2b(), blake2s(),
sha3_224, sha3_256, sha3_384, sha3_512, shake_128, and shake_256.

More algorithms may be available on your platform but the above are guaranteed
to exist.  See the algorithms_guaranteed and algorithms_available attributes
to find out what algorithm names can be passed to new().

NOTE: If you want the adler32 or crc32 hash functions they are available in
the zlib module.

Choose your hash function wisely.  Some have known collision weaknesses.
sha384 and sha512 will be slow on 32 bit platforms.

Hash objects have these methods:
 - update(data): Update the hash object with the bytes in data. Repeated calls
                 are equivalent to a single call with the concatenation of all
                 the arguments.
 - digest():     Return the digest of the bytes passed to the update() method
                 so far as a bytes object.
 - hexdigest():  Like digest() except the digest is returned as a string
                 of double length, containing only hexadecimal digits.
 - copy():       Return a copy (clone) of the hash object. This can be used to
                 efficiently compute the digests of datas that share a common
                 initial substring.

For example, to obtain the digest of the byte string 'Nobody inspects the
spammish repetition':

    >>> import hashlib
    >>> m = hashlib.md5()
    >>> m.update(b"Nobody inspects")
    >>> m.update(b" the spammish repetition")
    >>> m.digest()
    b'\xbbd\x9c\x83\xdd\x1e\xa5\xc9\xd9\xde\xc9\xa1\x8d\xf0\xff\xe9'

More condensed:

    >>> hashlib.sha224(b"Nobody inspects the spammish repetition").hexdigest()
    'a4337bc45a8fc544c03f52dc550cd6e1e87021bc896588bd79e901e2'
"""

import sys
from _typeshed import ReadableBuffer
from collections.abc import Callable, Set as AbstractSet
from typing import Protocol, final
from typing_extensions import Self

if sys.version_info >= (3, 11):
    __all__ = (
        "md5",
        "sha1",
        "sha224",
        "sha256",
        "sha384",
        "sha512",
        "blake2b",
        "blake2s",
        "sha3_224",
        "sha3_256",
        "sha3_384",
        "sha3_512",
        "shake_128",
        "shake_256",
        "new",
        "algorithms_guaranteed",
        "algorithms_available",
        "pbkdf2_hmac",
        "file_digest",
    )
else:
    __all__ = (
        "md5",
        "sha1",
        "sha224",
        "sha256",
        "sha384",
        "sha512",
        "blake2b",
        "blake2s",
        "sha3_224",
        "sha3_256",
        "sha3_384",
        "sha3_512",
        "shake_128",
        "shake_256",
        "new",
        "algorithms_guaranteed",
        "algorithms_available",
        "pbkdf2_hmac",
    )

class _Hash:
    @property
    def digest_size(self) -> int: ...
    @property
    def block_size(self) -> int: ...
    @property
    def name(self) -> str: ...
    def __init__(self, data: ReadableBuffer = ...) -> None: ...
    def copy(self) -> Self: ...
    def digest(self) -> bytes: ...
    def hexdigest(self) -> str: ...
    def update(self, data: ReadableBuffer, /) -> None: ...

if sys.version_info >= (3, 9):
    def new(name: str, data: ReadableBuffer = b"", *, usedforsecurity: bool = ...) -> _Hash:
        """
        new(name, data=b'') - Return a new hashing object using the named algorithm;
        optionally initialized with data (which must be a bytes-like object).
        """
        ...
    def md5(string: ReadableBuffer = b"", *, usedforsecurity: bool = True) -> _Hash:
        """Returns a md5 hash object; optionally initialized with a string"""
        ...
    def sha1(string: ReadableBuffer = b"", *, usedforsecurity: bool = True) -> _Hash:
        """Returns a sha1 hash object; optionally initialized with a string"""
        ...
    def sha224(string: ReadableBuffer = b"", *, usedforsecurity: bool = True) -> _Hash:
        """Returns a sha224 hash object; optionally initialized with a string"""
        ...
    def sha256(string: ReadableBuffer = b"", *, usedforsecurity: bool = True) -> _Hash:
        """Returns a sha256 hash object; optionally initialized with a string"""
        ...
    def sha384(string: ReadableBuffer = b"", *, usedforsecurity: bool = True) -> _Hash:
        """Returns a sha384 hash object; optionally initialized with a string"""
        ...
    def sha512(string: ReadableBuffer = b"", *, usedforsecurity: bool = True) -> _Hash:
        """Returns a sha512 hash object; optionally initialized with a string"""
        ...

else:
    def new(name: str, data: ReadableBuffer = b"") -> _Hash: ...
    def md5(string: ReadableBuffer = b"") -> _Hash: ...
    def sha1(string: ReadableBuffer = b"") -> _Hash: ...
    def sha224(string: ReadableBuffer = b"") -> _Hash: ...
    def sha256(string: ReadableBuffer = b"") -> _Hash: ...
    def sha384(string: ReadableBuffer = b"") -> _Hash: ...
    def sha512(string: ReadableBuffer = b"") -> _Hash: ...

algorithms_guaranteed: AbstractSet[str]
algorithms_available: AbstractSet[str]

def pbkdf2_hmac(
    hash_name: str, password: ReadableBuffer, salt: ReadableBuffer, iterations: int, dklen: int | None = None
) -> bytes:
    """Password based key derivation function 2 (PKCS #5 v2.0) with HMAC as pseudorandom function."""
    ...

class _VarLenHash:
    digest_size: int
    block_size: int
    name: str
    def __init__(self, data: ReadableBuffer = ...) -> None: ...
    def copy(self) -> _VarLenHash: ...
    def digest(self, length: int, /) -> bytes: ...
    def hexdigest(self, length: int, /) -> str: ...
    def update(self, data: ReadableBuffer, /) -> None: ...

sha3_224 = _Hash
sha3_256 = _Hash
sha3_384 = _Hash
sha3_512 = _Hash
shake_128 = _VarLenHash
shake_256 = _VarLenHash

def scrypt(
    password: ReadableBuffer, *, salt: ReadableBuffer, n: int, r: int, p: int, maxmem: int = 0, dklen: int = 64
) -> bytes:
    """scrypt password-based key derivation function."""
    ...
@final
class _BlakeHash(_Hash):
    MAX_DIGEST_SIZE: int
    MAX_KEY_SIZE: int
    PERSON_SIZE: int
    SALT_SIZE: int

    if sys.version_info >= (3, 9):
        def __init__(
            self,
            data: ReadableBuffer = ...,
            /,
            *,
            digest_size: int = ...,
            key: ReadableBuffer = ...,
            salt: ReadableBuffer = ...,
            person: ReadableBuffer = ...,
            fanout: int = ...,
            depth: int = ...,
            leaf_size: int = ...,
            node_offset: int = ...,
            node_depth: int = ...,
            inner_size: int = ...,
            last_node: bool = ...,
            usedforsecurity: bool = ...,
        ) -> None: ...
    else:
        def __init__(
            self,
            data: ReadableBuffer = ...,
            /,
            *,
            digest_size: int = ...,
            key: ReadableBuffer = ...,
            salt: ReadableBuffer = ...,
            person: ReadableBuffer = ...,
            fanout: int = ...,
            depth: int = ...,
            leaf_size: int = ...,
            node_offset: int = ...,
            node_depth: int = ...,
            inner_size: int = ...,
            last_node: bool = ...,
        ) -> None: ...

blake2b = _BlakeHash
blake2s = _BlakeHash

if sys.version_info >= (3, 11):
    class _BytesIOLike(Protocol):
        def getbuffer(self) -> ReadableBuffer: ...

    class _FileDigestFileObj(Protocol):
        def readinto(self, buf: bytearray, /) -> int: ...
        def readable(self) -> bool: ...

    def file_digest(
        fileobj: _BytesIOLike | _FileDigestFileObj, digest: str | Callable[[], _Hash], /, *, _bufsize: int = 262144
    ) -> _Hash:
        """
        Hash the contents of a file-like object. Returns a digest object.

        *fileobj* must be a file-like object opened for reading in binary mode.
        It accepts file objects from open(), io.BytesIO(), and SocketIO objects.
        The function may bypass Python's I/O and use the file descriptor *fileno*
        directly.

        *digest* must either be a hash algorithm name as a *str*, a hash
        constructor, or a callable that returns a hash object.
        """
        ...
