# coding: utf8
import os
import re
import sys

import sublime
import sublime_plugin
import difflib
import tempfile
import subprocess

from fnmatch import fnmatch
import codecs

if sublime.platform() == "windows":
    from subprocess import Popen


class FileDiffMenuCommand(sublime_plugin.TextCommand):
    def is_visible(self, **kwargs):
        return get_setting('show_context_menu', True)

    def run(self, edit, cmd=None):
        # Individual menu items.
        CLIPBOARD   = {'text': 'Diff file with Clipboard',          'command' : 'file_diff_clipboard'}
        SELECTIONS  = {'text': 'Diff Selections',                   'command' : 'file_diff_selections'}
        SAVED       = {'text': 'Diff file with Saved',              'command' : 'file_diff_saved'}
        FILE        = {'text': u'Diff file with File in Project…',  'command' : 'file_diff_file'}
        TAB         = {'text': u'Diff file with Open Tab…',         'command' : 'file_diff_tab'}
        PREVIOUS    = {'text': 'Diff file with Previous Tab',       'command' : 'file_diff_previous'}

        menu_items = [CLIPBOARD, SELECTIONS, SAVED, FILE, TAB, PREVIOUS]

        non_empty_regions = len([region for region in self.view.sel() if not region.empty()])

        if non_empty_regions != 2:
            menu_items.remove(SELECTIONS)

        if non_empty_regions and non_empty_regions != 2:
            for item in menu_items:
                item['text'] = item['text'].replace('Diff file', 'Diff selection')

        if cmd is not None:
            for item in menu_items:
                item['text'] += ' (' + os.path.splitext(os.path.basename(cmd[0]))[0] + ')'

        if not (self.view.file_name() and self.view.is_dirty()):
            menu_items.remove(SAVED)

        def on_done(index):
            if index >= 0:
                self.view.run_command(menu_items[index]['command'], {'cmd': cmd})

        self.view.window().show_quick_panel([item['text'] for item in menu_items], on_done)


class FileDiffCommand(sublime_plugin.TextCommand):
    def diff_content(self, view):
        content = ''

        for region in view.sel():
            if region.empty():
                continue
            content += view.substr(region)

        if not content:
            content = view.substr(sublime.Region(0, view.size()))
        return content

    def prep_content(self, ab, file_name, default_name):
        content = ab.splitlines(True)
        if file_name is None:
            file_name = default_name
        content = [line.replace("\r\n", "\n").replace("\r", "\n") for line in content]

        trim_trailing_white_space_before_diff = get_setting('trim_trailing_white_space_before_diff', False)
        if trim_trailing_white_space_before_diff:
            content = [line.rstrip() for line in content]

        return (content, file_name)

    def run_diff(self, a, b, from_file, to_file, **options):
        external_diff_tool = options.get('cmd')

        if options.get('reverse'):
            from_file, to_file = to_file, from_file
            a, b = b, a

        (from_content, from_file) = self.prep_content(a, from_file, 'from_file')
        (to_content, to_file) = self.prep_content(b, to_file, 'to_file')

        context_lines = get_setting("context_lines", 3);
        if context_lines == "full":
            context_lines = sys.maxsize

        diffs = list(difflib.unified_diff(from_content, to_content, from_file, to_file, n=context_lines))

        if not diffs:
            self.view.show_popup('No Difference')

        else:
            external_command = external_diff_tool or get_setting('cmd')
            open_in_sublime = get_setting('open_in_sublime', not external_command)

            if external_command:
                self.diff_with_external(external_command, a, b, from_file, to_file, **options)

            if open_in_sublime:
                # fix diffs
                diffs = map(lambda line: (line and line[-1] == "\n") and line or line + "\n", diffs)
                self.diff_in_sublime(diffs)

    def diff_with_external(self, external_command, a, b, from_file=None, to_file=None, **options):
        try:
            from_file_on_disk = self.file_will_be_read_from_disk(from_file)
            to_file_on_disk = self.file_will_be_read_from_disk(to_file)

            # If a dirty file is diffed against the copy on disk, we statically set
            # `from_file_on_disk` to True; so that the diff becomes "file to temp"
            # instead of "temp to temp".
            if to_file == from_file + " (Unsaved)":
                view = self.view.window().find_open_file(from_file)
                if os.path.exists(from_file) and view and view.is_dirty():
                    from_file_on_disk = True

            files_to_remove = []

            if not from_file_on_disk:
                tmp_file = tempfile.NamedTemporaryFile(dir = sublime.packages_path(), prefix = "file-diffs-", suffix = ".temp", delete=False)
                from_file = tmp_file.name
                tmp_file.close()
                files_to_remove.append(tmp_file.name)

                with codecs.open(from_file, encoding='utf-8', mode='w+') as tmp_file:
                    tmp_file.write(a)

            if not to_file_on_disk:
                tmp_file = tempfile.NamedTemporaryFile(dir = sublime.packages_path(), prefix = "file-diffs-", suffix = ".temp", delete=False)
                to_file = tmp_file.name
                tmp_file.close()
                files_to_remove.append(tmp_file.name)

                with codecs.open(to_file, encoding='utf-8', mode='w+') as tmp_file:
                    tmp_file.write(b)

            trim_trailing_white_space_before_diff = get_setting('trim_trailing_white_space_before_diff', False)
            if trim_trailing_white_space_before_diff:
                def trim_trailing_white_space(file_name):
                    trim_lines = []
                    modified = False
                    with codecs.open(file_name, encoding='utf-8', mode='r') as f:
                        lines = f.readlines()
                        lines = [line.replace("\n", '').replace("\r", '') for line in lines]
                        for line in lines:
                            trim_line = line.rstrip()
                            trim_lines.append(trim_line)
                            if trim_line != line:
                                modified = True
                    if modified:
                        tmp_file = tempfile.NamedTemporaryFile(dir = sublime.packages_path(), prefix = "file-diffs-", suffix = ".temp", delete=False)
                        file_name = tmp_file.name
                        tmp_file.close()
                        files_to_remove.append(tmp_file.name)
                        with codecs.open(file_name, encoding='utf-8', mode='w+') as f:
                            f.writelines('\n'.join(trim_lines))
                    return file_name

                from_file = trim_trailing_white_space(from_file)
                to_file = trim_trailing_white_space(to_file)

            if os.path.exists(from_file):
                external_command = [c.replace('$file1', from_file) for c in external_command]
                external_command = [c.replace('$file2', to_file) for c in external_command]
                external_command = [os.path.expandvars(c) for c in external_command]
                if sublime.platform() == "windows":
                    Popen(external_command)
                else:
                    subprocess.Popen(external_command)

                apply_tempfile_changes_after_diff_tool = get_setting('apply_tempfile_changes_after_diff_tool', False)
                post_diff_tool = options.get('post_diff_tool')
                if apply_tempfile_changes_after_diff_tool and post_diff_tool is not None and (not from_file_on_disk or not to_file_on_disk):
                    if from_file_on_disk:
                        from_file = None
                    if to_file_on_disk:
                        to_file = None
                    # Use a dialog to block st and wait for the closing of the diff tool
                    if sublime.ok_cancel_dialog("Apply changes from tempfile after external diff tool execution?"):
                        post_diff_tool(from_file, to_file)
        except Exception as e:
            # some basic logging here, since we are cluttering the /tmp folder
            self.view.show_popup(str(e))

        finally:
            def remove_files():
                for file in files_to_remove:
                    os.remove(file)

            # Remove temp files after 15 seconds. We don't remove immediately,
            # because external diff tools may take some time to read them from disk.
            sublime.set_timeout_async(remove_files, 15000)

    def diff_in_sublime(self, diffs):
        diffs = ''.join(diffs)
        global scratches
        global do_not_record
        do_not_record = True
        scratch = self.view.window().new_file()
        scratches.add(scratch.id())
        scratch.set_scratch(True)
        scratch.set_syntax_file('Packages/Diff/Diff.tmLanguage')
        scratch.run_command('file_diff_dummy1', {'content': diffs})
        do_not_record = False

    def read_file(self, file_name):
        content = ''
        with codecs.open(file_name, mode='U', encoding='utf-8') as f:
            content = f.read()
        return content

    def get_file_name(self, view, default_name):
        file_name = ''
        if view.file_name():
            file_name = view.file_name()
        elif view.name():
            file_name = view.name()
        else:
            file_name = default_name
        return file_name

    def get_content_from_file(self, file_name):
        with codecs.open(file_name, encoding='utf-8', mode='r') as f:
            lines = f.readlines()
            lines = [line.replace("\r\n", "\n").replace("\r", "\n") for line in lines]
            content = ''.join(lines)
            return content

    def update_view(self, view, edit, tmp_file):
        if tmp_file:
            non_empty_regions = [region for region in view.sel() if not region.empty()]
            nb_non_empty_regions = len(non_empty_regions)
            region = None
            if nb_non_empty_regions == 0:
                region = sublime.Region(0, view.size())
            elif nb_non_empty_regions == 1:
                region = non_empty_regions[0]
            else:
                self.view.show_popup('Cannot update multiselection')
                return
            view.replace(edit, region, self.get_content_from_file(tmp_file))

    def file_will_be_read_from_disk(self, file):
        view = self.view.window().find_open_file(file)
        return os.path.exists(file) and not (view and view.is_dirty()) and not (view and self.view_has_a_selection(view))

    def view_has_a_selection(self, view):
        """Checks if it has exactly one non-empty selection."""
        return len(view.sel()) == 1 and not view.sel()[0].empty()


class FileDiffDummy1Command(sublime_plugin.TextCommand):
    def run(self, edit, content):
        self.view.insert(edit, 0, content)


class FileDiffClipboardCommand(FileDiffCommand):
    def run(self, edit, **kwargs):
        to_file = self.get_file_name(self.view, 'untitled')
        for region in self.view.sel():
            if not region.empty():
                to_file += ' (Selection)'
                break
        clipboard = sublime.get_clipboard()
        def on_post_diff_tool(from_file, to_file):
            self.update_view(self.view, edit, to_file)
            sublime.set_clipboard(self.get_content_from_file(from_file))

        reverse = kwargs.get('reverse') or get_setting('reverse_clipboard', False)
        kwargs.update({'post_diff_tool': on_post_diff_tool, 'reverse': reverse})

        self.run_diff(clipboard, self.diff_content(self.view),
            from_file='(clipboard)',
            to_file=to_file,
            **kwargs)

    def is_visible(self):
        if not get_setting('show_context_menu', True):
            return False
        return bool(sublime.get_clipboard())


class FileDiffSelectionsCommand(FileDiffCommand):
    def trim_indent(self, lines):
        indent = None
        for line in lines:
            # ignore blank lines
            if not line:
                continue

            new_indent = re.match('[ \t]*', line).group(0)
            # ignore lines that only consist of whitespace
            if len(new_indent) == len(line):
                continue

            if indent is None:
                indent = new_indent
            elif len(new_indent) < len(indent):
                indent = new_indent

            if not indent:
                break
        return indent

    def run(self, edit, **kwargs):
        regions = self.view.sel()
        first_selection = self.view.substr(regions[0])
        second_selection = self.view.substr(regions[1])

        # trim off indent
        indent = self.trim_indent(first_selection.splitlines())
        if indent:
            first_selection = "\n".join(line[len(indent):] for line in first_selection.splitlines())

        # trim off indent
        indent = self.trim_indent(second_selection.splitlines())
        if indent:
            second_selection = "\n".join(line[len(indent):] for line in second_selection.splitlines())

        self.run_diff(first_selection, second_selection,
            from_file='first selection',
            to_file='second selection',
            **kwargs)

    def is_visible(self):
        if not get_setting('show_context_menu', True):
            return False
        return len(self.view.sel()) > 1


class FileDiffSavedCommand(FileDiffCommand):
    def run(self, edit, **kwargs):
        def on_post_diff_tool(from_file, to_file):
            self.update_view(self.view, edit, to_file)

        kwargs.update({'post_diff_tool': on_post_diff_tool})
        self.run_diff(self.read_file(self.view.file_name()), self.diff_content(self.view),
            from_file=self.view.file_name(),
            to_file=self.view.file_name() + ' (Unsaved)',
            **kwargs)

    def is_visible(self):
        if not get_setting('show_context_menu', True):
            return False
        return bool(self.view.file_name()) and self.view.is_dirty()


class FileDiffFileCommand(FileDiffCommand):
    def run(self, edit, **kwargs):
        common = None
        folders = self.view.window().folders()
        files = self.find_files(folders, [])
        for folder in folders:
            if common is None:
                common = folder
            else:
                common_len = len(common)
                while folder[0:common_len] != common[0:common_len]:
                    common_len -= 1
                    common = common[0:common_len]

        my_file = self.view.file_name()
        # filter out my_file
        files = [f for f in files if f != my_file]
        # shorten names using common length
        file_picker = [f[len(common):] for f in files]

        def on_done(index):
            if index > -1:
                self.run_diff(self.diff_content(self.view), self.read_file(files[index]),
                    from_file=self.view.file_name(),
                    to_file=files[index],
                    **kwargs)
        sublime.set_timeout(lambda: self.view.window().show_quick_panel(file_picker, on_done), 1)

    def find_files(self, folders, ret=[]):
        # Cannot access these settings!!  WHY!?
        # folder_exclude_patterns = self.view.get_setting('folder_exclude_patterns')
        # file_exclude_patterns = self.view.get_setting('file_exclude_patterns')
        folder_exclude_patterns = [".svn", ".git", ".hg", "CVS"]
        file_exclude_patterns = ["*.pyc", "*.pyo", "*.exe", "*.dll", "*.obj", "*.o", "*.a", "*.lib", "*.so", "*.dylib", "*.ncb", "*.sdf", "*.suo", "*.pdb", "*.idb", ".DS_Store", "*.class", "*.psd", "*.db"]
        max_files = get_setting('limit', 1000)

        for folder in folders:
            if not os.path.isdir(folder):
                continue

            for f in os.listdir(folder):
                fullpath = os.path.join(folder, f)
                if os.path.isdir(fullpath):
                    # excluded folder?
                    if not len([True for pattern in folder_exclude_patterns if fnmatch(f, pattern)]):
                        self.find_files([fullpath], ret)
                else:
                    # excluded file?
                    if not len([True for pattern in file_exclude_patterns if fnmatch(f, pattern)]):
                        ret.append(fullpath)
                if len(ret) >= max_files:
                    self.view.show_popup('Too many files to include all of them in this list')
                    return ret
        return ret

    def is_visible(self, **kwargs):
        return get_setting('show_context_menu', True)


class FileDiffTabCommand(FileDiffCommand):
    def run(self, edit, **kwargs):
        my_id = self.view.id()
        files = []
        contents = []
        views = []
        untitled_count = 1
        for v in self.view.window().views():
            if v.id() != my_id:
                this_content = v.substr(sublime.Region(0, v.size()))
                if v.file_name():
                    files.append(v.file_name())
                elif v.name():
                    files.append(v.name())
                else:
                    files.append('untitled %d' % untitled_count)
                    untitled_count += 1

                contents.append(this_content)
                views.append(v)

        def on_done(index):
            if index > -1:
                def on_post_diff_tool(from_file, to_file):
                    self.update_view(self.view, edit, from_file)
                    self.update_view(views[index], edit, to_file)

                kwargs.update({'post_diff_tool': on_post_diff_tool})
                self.run_diff(self.diff_content(self.view), contents[index],
                    from_file=self.view.file_name(),
                    to_file=files[index],
                    **kwargs)

        if len(files) == 1:
            on_done(0)
        else:
            if get_setting('expand_full_file_name_in_tab', False):
                menu_items = [[os.path.basename(f),f] for f in files]
            else:
                menu_items = [os.path.basename(f) for f in files]
            sublime.set_timeout(lambda: self.view.window().show_quick_panel(menu_items, on_done), 1)

    def is_visible(self):
        if not get_setting('show_context_menu', True):
            return False
        return len(self.view.window().views()) > 1


previous_view = current_view = None
do_not_record = False
scratches = set()

class FileDiffPreviousCommand(FileDiffCommand):
    def run(self, edit, **kwargs):
        if previous_view:
            def on_post_diff_tool(from_file, to_file):
                self.update_view(previous_view, edit, from_file)
                self.update_view(current_view, edit, to_file)

            kwargs.update({'post_diff_tool': on_post_diff_tool})
            self.run_diff(self.diff_content(previous_view), self.diff_content(self.view),
                from_file=self.get_file_name(previous_view, 'untitled (Previous)'),
                to_file=self.get_file_name(self.view, 'untitled (Current)'),
                **kwargs)

    def is_visible(self):
        if not get_setting('show_context_menu', True):
            return False
        return previous_view is not None

def record_current_view(view):
    if do_not_record:
        return
    settings = view.settings()
    if view.id() in scratches:
        return
    global previous_view
    global current_view
    previous_view = current_view
    current_view = view

class FileDiffListener(sublime_plugin.EventListener):
    def on_activated(self, view):
        try:
            # Prevent 'show_quick_panel()' of 'FileDiffs Menu' from being recorded
            viewids = [v.id() for v in view.window().views()]
            if view.id() not in viewids:
                return
            if current_view is None or view.id() != current_view.id():
                record_current_view(view)
        except AttributeError:
            pass

    def on_close(self, view):
        if view.id() in scratches:
            scratches.remove(view.id())


def get_setting(key, default=None):
    settings = sublime.load_settings('FileDiffs.sublime-settings')
    os_specific_settings = {}
    if sublime.platform() == 'windows':
        os_specific_settings = sublime.load_settings('FileDiffs (Windows).sublime-settings')
    elif sublime.platform() == 'osx':
        os_specific_settings = sublime.load_settings('FileDiffs (OSX).sublime-settings')
    else:
        os_specific_settings = sublime.load_settings('FileDiffs (Linux).sublime-settings')
    return os_specific_settings.get(key, settings.get(key, default))
