"""Porter Duff compositing."""
from __future__ import annotations
from abc import ABCMeta, abstractmethod


class PorterDuff(metaclass=ABCMeta):
    """Porter Duff compositing."""

    def __init__(self, cba: float, csa: float) -> None:
        """Initialize."""

        self.cba = cba
        self.csa = csa

    @abstractmethod
    def fa(self) -> float:  # pragma: no cover
        """Calculate `Fa`."""

        raise NotImplementedError('fa is not implemented')

    @abstractmethod
    def fb(self) -> float:  # pragma: no cover
        """Calculate `Fb`."""

        raise NotImplementedError('fb is not implemented')

    def co(self, cb: float, cs: float) -> float:
        """Calculate premultiplied coordinate."""

        return self.csa * self.fa() * cs + self.cba * self.fb() * cb

    def ao(self) -> float:
        """Calculate output alpha."""

        return self.csa * self.fa() + self.cba * self.fb()


class Clear(PorterDuff):
    """Clear."""

    def fa(self) -> float:
        """Calculate `Fa`."""

        return 0.0

    def fb(self) -> float:
        """Calculate `Fb`."""

        return 0.0


class Copy(PorterDuff):
    """Copy."""

    def fa(self) -> float:
        """Calculate `Fa`."""

        return 1.0

    def fb(self) -> float:
        """Calculate `Fb`."""

        return 0.0


class Destination(PorterDuff):
    """Destination."""

    def fa(self) -> float:
        """Calculate `Fa`."""

        return 0.0

    def fb(self) -> float:
        """Calculate `Fb`."""

        return 1.0


class SourceOver(PorterDuff):
    """Source over."""

    def fa(self) -> float:
        """Calculate `Fa`."""

        return 1.0

    def fb(self) -> float:
        """Calculate `Fb`."""

        return 1.0 - self.csa


class DestinationOver(PorterDuff):
    """Destination over."""

    def fa(self) -> float:
        """Calculate `Fa`."""

        return 1.0 - self.cba

    def fb(self) -> float:
        """Calculate `Fb`."""

        return 1.0


class SourceIn(PorterDuff):
    """Source in."""

    def fa(self) -> float:
        """Calculate `Fa`."""

        return self.cba

    def fb(self) -> float:
        """Calculate `Fb`."""

        return 0.0


class DestinationeIn(PorterDuff):
    """Destination in."""

    def fa(self) -> float:
        """Calculate `Fa`."""

        return 0.0

    def fb(self) -> float:
        """Calculate `Fb`."""

        return self.csa


class SourceOut(PorterDuff):
    """Source out."""

    def fa(self) -> float:
        """Calculate `Fa`."""

        return 1.0 - self.cba

    def fb(self) -> float:
        """Calculate `Fb`."""

        return 0.0


class DestinationOut(PorterDuff):
    """Destination out."""

    def fa(self) -> float:
        """Calculate `Fa`."""

        return 0.0

    def fb(self) -> float:
        """Calculate `Fb`."""

        return 1.0 - self.csa


class SourceAtop(PorterDuff):
    """Source atop."""

    def fa(self) -> float:
        """Calculate `Fa`."""

        return self.cba

    def fb(self) -> float:
        """Calculate `Fb`."""

        return 1.0 - self.csa


class DestinationAtop(PorterDuff):
    """Destination atop."""

    def fa(self) -> float:
        """Calculate `Fa`."""

        return 1.0 - self.cba

    def fb(self) -> float:
        """Calculate `Fb`."""

        return self.csa


class XOR(PorterDuff):
    """XOR."""

    def fa(self) -> float:
        """Calculate `Fa`."""

        return 1.0 - self.cba

    def fb(self) -> float:
        """Calculate `Fb`."""

        return 1.0 - self.csa


class Lighter(PorterDuff):
    """Lighter."""

    def fa(self) -> float:
        """Calculate `Fa`."""

        return 1.0

    def fb(self) -> float:
        """Calculate `Fb`."""

        return 1.0


class PlusDarker(PorterDuff):
    """Plus darker."""

    def fa(self) -> float:
        """Calculate `Fa`."""

        return 1.0

    def fb(self) -> float:
        """Calculate `Fb`."""

        return 1.0

    def co(self, cb: float, cs: float) -> float:
        """Calculate premultiplied coordinate."""

        return self.ao() - min(1.0, super().co(1.0 - cb, 1.0 - cs))

    def ao(self) -> float:
        """Calculate output alpha."""

        return min(1.0, super().ao())


class PlusLigher(PorterDuff):
    """Plus lighter."""

    def fa(self) -> float:
        """Calculate `Fa`."""

        return 1.0

    def fb(self) -> float:
        """Calculate `Fb`."""

        return 1.0

    def co(self, cb: float, cs: float) -> float:
        """Calculate premultiplied coordinate."""

        return min(1.0, super().co(cb, cs))

    def ao(self) -> float:
        """Calculate output alpha."""

        return min(1.0, super().ao())


SUPPORTED = {
    'clear': Clear,
    'copy': Copy,
    'destination': Destination,
    'source-over': SourceOver,
    'destination-over': DestinationOver,
    'source-in': SourceIn,
    'destination-in': DestinationeIn,
    'source-out': SourceOut,
    'destination-out': DestinationOut,
    'source-atop': SourceAtop,
    'destination-atop': DestinationAtop,
    'xor': XOR,
    'lighter': Lighter,
    'plus-darker': PlusDarker,
    'plus-lighter': PlusLigher
}


def compositor(name: str) -> type[PorterDuff]:
    """Get the requested compositor."""

    composite = SUPPORTED.get(name)
    if not composite:
        raise ValueError(f"'{name}' compositing is not supported")
    return composite
