"""Colors."""
from __future__ import annotations
import sys
import abc
import functools
import random
import math
from . import cat
from . import distance
from . import convert
from . import gamut
from . import compositing
from . import interpolate
from . import filters
from . import contrast
from . import harmonies
from . import average
from . import temperature
from . import util
from . import algebra as alg
from .channels import ANGLE_DEG, ANGLE_RAD, ANGLE_GRAD, ANGLE_TURN, ANGLE_NULL
from .deprecate import warn_deprecated
from itertools import zip_longest as zipl
from .css import parse
from .types import VectorLike, Vector, ColorInput
from .spaces import Space, RGBish
from .spaces.hsv import HSV
from .spaces.srgb.css import sRGB
from .spaces.srgb_linear import sRGBLinear
from .spaces.hsl.css import HSL
from .spaces.hwb.css import HWB
from .spaces.lab.css import Lab
from .spaces.lch.css import LCh
from .spaces.lab_d65 import LabD65
from .spaces.lch_d65 import LChD65
from .spaces.display_p3 import DisplayP3
from .spaces.display_p3_linear import DisplayP3Linear
from .spaces.a98_rgb import A98RGB
from .spaces.a98_rgb_linear import A98RGBLinear
from .spaces.prophoto_rgb import ProPhotoRGB
from .spaces.prophoto_rgb_linear import ProPhotoRGBLinear
from .spaces.rec2020 import Rec2020
from .spaces.rec2020_linear import Rec2020Linear
from .spaces.xyz_d65 import XYZD65
from .spaces.xyz_d50 import XYZD50
from .spaces.oklab.css import Oklab
from .spaces.oklch.css import OkLCh
from .spaces.rec2100_pq import Rec2100PQ
from .spaces.rec2100_hlg import Rec2100HLG
from .spaces.rec2100_linear import Rec2100Linear
from .spaces.jzazbz.css import Jzazbz
from .spaces.jzczhz.css import JzCzhz
from .spaces.ictcp.css import ICtCp
from .distance import DeltaE
from .distance.delta_e_76 import DE76
from .distance.delta_e_94 import DE94
from .distance.delta_e_cmc import DECMC
from .distance.delta_e_2000 import DE2000
from .distance.delta_e_hyab import DEHyAB
from .distance.delta_e_ok import DEOK
from .distance.delta_e_itp import DEITP
from .distance.delta_e_z import DEZ
from .contrast import ColorContrast
from .contrast.wcag21 import WCAG21Contrast
from .gamut import Fit
from .gamut.fit_minde_chroma import MINDEChroma
from .gamut.fit_lch_chroma import LChChroma
from .gamut.fit_oklch_chroma import OkLChChroma
from .gamut.fit_raytrace import RayTrace
from .cat import CAT, Bradford
from .filters import Filter
from .filters.w3c_filter_effects import Sepia, Brightness, Contrast, Saturate, Opacity, HueRotate, Grayscale, Invert
from .filters.cvd import Protan, Deutan, Tritan
from .interpolate import Interpolator, Interpolate
from .interpolate.linear import Linear
from .interpolate.css_linear import CSSLinear
from .interpolate.continuous import Continuous
from .interpolate.bspline import BSpline
from .interpolate.bspline_natural import NaturalBSpline
from .interpolate.monotone import Monotone
from .temperature import CCT
from .temperature.ohno_2013 import Ohno2013
from .temperature.robertson_1968 import Robertson1968
from .types import Plugin
from typing import Iterator, overload, Sequence, Iterable, Any, Callable, Mapping
if (3, 11) <= sys.version_info:
    from typing import Self
else:
    from typing_extensions import Self

SUPPORTED_CHROMATICITY_SPACES = {'xyz', 'uv-1960', 'uv-1976', 'xy-1931'}

POSTFIX = {
    ANGLE_NULL: '',
    ANGLE_DEG: 'deg',
    ANGLE_RAD: 'rad',
    ANGLE_GRAD: 'grad',
    ANGLE_TURN: 'trun'
}


class ColorMatch:
    """Color match object."""

    __slots__ = ('color', 'start', 'end')

    def __init__(self, color: Color, start: int, end: int) -> None:
        """Initialize."""

        self.color = color
        self.start = start
        self.end = end

    def __str__(self) -> str:  # pragma: no cover
        """String."""

        return f"ColorMatch(color={self.color!r}, start={self.start}, end={self.end})"

    __repr__ = __str__


class ColorMeta(abc.ABCMeta):
    """Ensure on subclass that the subclass has new instances of mappings."""

    def __init__(cls, name: str, bases: tuple[object, ...], clsdict: dict[str, Any]) -> None:
        """Copy mappings on subclass."""

        # Ensure subclassed Color objects do not use the same plugin mappings
        if len(cls.mro()) > 2:
            cls.CS_MAP = cls.CS_MAP.copy()  # type: dict[str, Space]
            cls.DE_MAP = cls.DE_MAP.copy()  # type: dict[str, DeltaE]
            cls.FIT_MAP = cls.FIT_MAP.copy()  # type: dict[str, Fit]
            cls.CAT_MAP = cls.CAT_MAP.copy()  # type: dict[str, CAT]
            cls.FILTER_MAP = cls.FILTER_MAP.copy()  # type: dict[str, Filter]
            cls.CONTRAST_MAP = cls.CONTRAST_MAP.copy()  # type: dict[str, ColorContrast]
            cls.INTERPOLATE_MAP = cls.INTERPOLATE_MAP.copy()  # type: dict[str, Interpolate[Any]]
            cls.CCT_MAP = cls.CCT_MAP.copy()  # type: dict[str, CCT]

        # Ensure each derived class tracks its own conversion paths for color spaces
        # relative to the installed color space plugins.
        @classmethod  # type: ignore[misc]
        @functools.lru_cache(maxsize=256)
        def _get_convert_chain(
            cls: type[Color],
            space: Space,
            target: str
        ) -> list[tuple[Space, Space, int, bool]]:
            """Resolve a conversion chain, cache it for speed."""

            return convert.get_convert_chain(cls, space, target)

        cls._get_convert_chain = _get_convert_chain


class Color(metaclass=ColorMeta):
    """Color class object which provides access and manipulation of color spaces."""

    CS_MAP = {}  # type: dict[str, Space]
    DE_MAP = {}  # type: dict[str, DeltaE]
    FIT_MAP = {}  # type: dict[str, Fit]
    CAT_MAP = {}  # type: dict[str, CAT]
    CONTRAST_MAP = {}  # type: dict[str, ColorContrast]
    FILTER_MAP = {}  # type: dict[str, Filter]
    INTERPOLATE_MAP = {}  # type: dict[str, Interpolate[Self]]
    CCT_MAP = {}  # type: dict[str, CCT]
    PRECISION = util.DEF_PREC
    ROUNDING = util.DEF_ROUND_MODE
    FIT = util.DEF_FIT
    INTERPOLATE = util.DEF_INTERPOLATE
    INTERPOLATOR = util.DEF_INTERPOLATOR
    DELTA_E = util.DEF_DELTA_E
    HARMONY = util.DEF_HARMONY
    AVERAGE = util.DEF_AVERAGE
    CHROMATIC_ADAPTATION = util.DEF_CHROMATIC_ADAPTATION
    CONTRAST = util.DEF_CONTRAST
    CCT = util.DEF_CCT
    POWERLESS = False
    CARRYFORWARD = False

    # It is highly unlikely that a user would ever need to override this, but
    # just in case, it is exposed, but undocumented.
    #
    # This is meant to prevent infinite loops in the event that a user registers
    # poorly crafted color spaces with circular convert linkage or somehow doesn't
    # resolve to XYZ. 10 is a generous size as our current largest iteration chain
    # is 6, and increasing that past 10 seems highly unlikely:
    #    XYZ -> sRGB Linear -> sRGB -> HSL -> HSV -> HWB
    _MAX_CONVERT_ITERATIONS = 10

    def __init__(
        self,
        color: ColorInput,
        data: VectorLike | None = None,
        alpha: float = util.DEF_ALPHA,
        **kwargs: Any
    ) -> None:
        """Initialize."""

        self._space, self._coords = self._parse(color, data, alpha, **kwargs)

    def __len__(self) -> int:
        """Get number of channels."""

        return len(self._space.channels)

    def __iter__(self) -> Iterator[float]:
        """Initialize iterator."""

        return iter(self._coords)

    @overload
    def __getitem__(self, i: str | int) -> float:
        ...

    @overload
    def __getitem__(self, i: slice) -> Vector:
        ...

    def __getitem__(self, i: str | int | slice) -> float | Vector:
        """Get channels."""

        return self._coords[self._space.get_channel_index(i)] if isinstance(i, str) else self._coords[i]

    @overload
    def __setitem__(self, i: str | int, v: float) -> None:
        ...

    @overload
    def __setitem__(self, i: slice, v: Vector) -> None:
        ...

    def __setitem__(self, i: str | int | slice, v: float | Vector) -> None:
        """Set channels."""

        space = self._space
        if isinstance(i, slice):
            for index, value in zip(range(len(self._coords))[i], v):  # type: ignore[arg-type]
                self._coords[index] = space.channels[index].limit(value)
        else:
            index = space.get_channel_index(i) if isinstance(i, str) else i
            self._coords[index] = space.channels[index].limit(v)  # type: ignore[arg-type]

    def __eq__(self, other: Any) -> bool:
        """Compare equal."""

        return (
            type(other) is type(self) and
            other.space() == self.space() and
            util.cmp_coords(other[:], self[:])
        )

    @classmethod
    def _parse(
        cls,
        color: ColorInput,
        data: VectorLike | None = None,
        alpha: float = util.DEF_ALPHA,
        **kwargs: Any
    ) -> tuple[Space, Vector]:
        """Parse the color."""

        # Parse a color string or color space name and coordinates
        if isinstance(color, str):

            # Parse a color space name and coordinates
            if data is not None:
                space_class = cls.CS_MAP.get(color)
                if not space_class:
                    raise ValueError(f"'{color}' is not a registered color space")
                num_channels = len(space_class.CHANNELS)
                num_data = len(data)
                if num_data < num_channels:
                    data = [*data, *[math.nan] * (num_channels - num_data)]
                coords = [c.limit(v) for c, v in zipl(space_class.CHANNELS, data)]
                coords.append(space_class.channels[-1].limit(alpha))
                obj = space_class, coords

            # Parse a CSS string
            else:
                m = cls._match(color, fullmatch=True)
                if m is None:
                    raise ValueError(f"'{color}' is not a valid color")
                coords = [c.limit(v) for c, v in zipl(m[0].CHANNELS, m[1])]
                coords.append(m[0].channels[-1].limit(m[2]))
                obj = m[0], coords

        # Handle a color instance
        elif isinstance(color, Color):
            cs = color._space
            space_class = cls.CS_MAP.get(cs.NAME)
            if not space_class or type(cs) is not type(space_class):
                raise ValueError(f"{type(cs)} is not a registered color space within {cls}")
            obj = space_class, color[:]

        # Handle a color dictionary
        elif isinstance(color, Mapping):
            obj = cls._parse(color['space'], color['coords'], color.get('alpha', 1.0))

        else:
            raise TypeError(f"{type(color)} is an unrecognized type")

        return obj

    @classmethod
    def _match(
        cls,
        string: str,
        start: int = 0,
        fullmatch: bool = False
    ) -> tuple[Space, Vector, float, int, int] | None:
        """
        Match a color in a buffer and return a color object.

        This must return the color space, not the Color object.
        """

        # Attempt color match
        if string[start:start + 6].lower() == 'color(':
            for space_class in cls.CS_MAP.values():
                if not space_class.COLOR_FORMAT:  # pragma: no cover
                    continue
                m = parse.parse_css(space_class, string, start, fullmatch, True)
                if m is not None:
                    return space_class, m[0][0], m[0][1], start, m[1]

        # Attempt color space specific match
        for space_class in cls.CS_MAP.values():
            m2 = space_class.match(string, start, fullmatch)
            if m2 is not None:
                return space_class, m2[0][0], m2[0][1], start, m2[1]
        return None

    @classmethod
    def match(
        cls,
        string: str,
        start: int = 0,
        fullmatch: bool = False
    ) -> ColorMatch | None:
        """Match color."""

        m = cls._match(string, start, fullmatch)
        if m is not None:
            return ColorMatch(cls(m[0].NAME, m[1], m[2]), m[3], m[4])
        return None

    @classmethod
    def _is_this_color(cls, obj: Any) -> bool:
        """Test if the input is "this" Color, not a subclass."""

        return type(obj) is cls

    @classmethod
    def _is_color(cls, obj: Any) -> bool:
        """Test if the input is a Color."""

        return isinstance(obj, Color)

    @classmethod
    def register(
        cls,
        plugin: Plugin | Sequence[Plugin],
        *,
        overwrite: bool = False,
        silent: bool = False
    ) -> None:
        """Register the hook."""

        reset_convert_cache = False
        mapping = None  # type: Any
        p = None  # type: Any
        for i in [plugin] if not isinstance(plugin, Sequence) else plugin:
            if isinstance(i, Space):
                mapping = cls.CS_MAP
                reset_convert_cache = True
                p = i
            elif isinstance(i, DeltaE):
                mapping = cls.DE_MAP
                p = i
            elif isinstance(i, CAT):
                mapping = cls.CAT_MAP
                p = i
            elif isinstance(i, Filter):
                mapping = cls.FILTER_MAP
                p = i
            elif isinstance(i, ColorContrast):
                mapping = cls.CONTRAST_MAP
                p = i
            elif isinstance(i, Interpolate):
                mapping = cls.INTERPOLATE_MAP
                p = i
            elif isinstance(i, CCT):
                mapping = cls.CCT_MAP
                p = i
            elif isinstance(i, Fit):
                mapping = cls.FIT_MAP
                p = i
                if p.NAME == 'clip':
                    if reset_convert_cache:  # pragma: no cover
                        cls._get_convert_chain.cache_clear()
                    if not silent:
                        raise ValueError("'{}' is a reserved name for gamut mapping/reduction and cannot be overridden")
                    continue  # pragma: no cover
            else:
                if reset_convert_cache:  # pragma: no cover
                    cls._get_convert_chain.cache_clear()
                raise TypeError(f"Cannot register plugin of type '{type(i)}'")

            if p.NAME != "*" and (p.NAME not in mapping or overwrite):
                mapping[p.NAME] = p
            elif not silent:
                if reset_convert_cache:  # pragma: no cover
                    cls._get_convert_chain.cache_clear()
                raise ValueError(f"A plugin of name '{p.NAME}' already exists or is not allowed")

        if reset_convert_cache:
            cls._get_convert_chain.cache_clear()

    @classmethod
    def deregister(cls, plugin: str | Sequence[str], *, silent: bool = False) -> None:
        """Deregister a plugin by name of specified plugin type."""

        reset_convert_cache = False

        if isinstance(plugin, str):
            plugin = [plugin]

        mapping = None  # type: dict[str, Any] | None
        for p in plugin:
            if p == '*':
                cls.CS_MAP.clear()
                cls.DE_MAP.clear()
                cls.CAT_MAP.clear()
                cls.FILTER_MAP.clear()
                cls.CONTRAST_MAP.clear()
                cls.INTERPOLATE_MAP.clear()
                cls.CCT_MAP.clear()
                cls.FIT_MAP.clear()
                return

            ptype, name = p.split(':', 1)
            if ptype == 'space':
                mapping = cls.CS_MAP
                reset_convert_cache = True
            elif ptype == "delta-e":
                mapping = cls.DE_MAP
            elif ptype == 'cat':
                mapping = cls.CAT_MAP
            elif ptype == 'filter':
                mapping = cls.FILTER_MAP
            elif ptype == 'contrast':
                mapping = cls.CONTRAST_MAP
            elif ptype == 'interpolate':
                mapping = cls.INTERPOLATE_MAP
            elif ptype == 'cct':
                mapping = cls.CCT_MAP
            elif ptype == "fit":
                mapping = cls.FIT_MAP
                if name == 'clip':
                    if reset_convert_cache:  # pragma: no cover
                        cls._get_convert_chain.cache_clear()
                    if not silent:
                        raise ValueError(
                            f"'{name}' is a reserved name gamut mapping/reduction and cannot be removed"
                        )
                    continue  # pragma: no cover
            else:
                if reset_convert_cache:  # pragma: no cover
                    cls._get_convert_chain.cache_clear()
                raise ValueError(f"The plugin category of '{ptype}' is not recognized")

            if name == '*':
                mapping.clear()
            elif name in mapping:
                del mapping[name]
            elif not silent:
                if reset_convert_cache:
                    cls._get_convert_chain.cache_clear()
                raise ValueError(f"A plugin of name '{name}' under category '{ptype}' could not be found")

        if reset_convert_cache:
            cls._get_convert_chain.cache_clear()

    @classmethod
    def random(cls, space: str, *, limits: Sequence[Sequence[float] | None] | None = None) -> Self:
        """Get a random color."""

        # Get the color space and number of channels
        cs = cls.CS_MAP[space]
        num_chan = len(cs.CHANNELS)

        # Initialize constraints if none were provided
        if limits is None:
            limits = []

        # Acquire the minimum and maximum for the channel and get a random value between
        length = len(limits)
        coords = []
        for i in range(num_chan):
            chan = limits[i] if i < length else None  # type: Any
            if chan is None:
                chan = cs.channels[i]
                a, b = chan.low, chan.high
            else:
                a, b = chan

            coords.append(random.uniform(a, b))

        # Create the color
        obj = cls(space, coords)
        if obj._space.is_polar():
            obj.normalize()
        return obj

    @classmethod
    def blackbody(
        cls,
        space: str,
        temp: float,
        duv: float = 0.0,
        *,
        scale: bool = True,
        scale_space: str | None = None,
        method: str | None = None,
        **kwargs: Any
    ) -> Self:
        """
        Get a color along the black body curve.

        Colors are specified by temperature in Kelvin. Depending on the algorithm, the practical
        range may differ.

        Colors are returned and normalized to be within the specified RGB space (preferably linear).
        The returned color, after normalization, is an approximation and may not exactly match the
        temperature. If `space` is set to `None` the color will not be normalized and may not be in
        the visible spectrum, but it should correlate with the specified temperature assuming it is
        not too far from the locus.
        """

        cct = temperature.cct(method, cls)
        color = cct.from_cct(cls, space, temp, duv, scale, scale_space, **kwargs)
        return color

    def cct(self, *, method: str | None = None, **kwargs: Any) -> Vector:
        """Get color temperature."""

        cct = temperature.cct(method, self)
        return cct.to_cct(self, **kwargs)

    def to_dict(
        self,
        *,
        nans: bool = True,
        precision: int | Sequence[int] | None = None,
        rounding: str | None = None
    ) -> Mapping[str, Any]:
        """Return color as a data object."""

        if precision is None or isinstance(precision, int):
            precision_alpha = precision
        else:
            precision_alpha = util.get_index(precision, len(self._space.channels) - 1, self.PRECISION)

        return {
            'space': self.space(),
            'coords': self.coords(nans=nans, precision=precision, rounding=rounding),
            'alpha': self.alpha(nans=nans, precision=precision_alpha, rounding=rounding)
        }

    def normalize(self, *, nans: bool = True) -> Self:
        """Normalize the color."""

        self[:-1] = self._space.normalize(self.coords(nans=False))
        if nans and self._space.is_polar() and self.is_achromatic():
            i = self._space.hue_index()  # type: ignore[attr-defined]
            self[i] = math.nan
        alpha = self[-1]
        self[-1] = 0.0 if math.isnan(alpha) else alpha
        return self

    def is_nan(self, name: str) -> bool:  # pragma: no cover
        """Check if channel is NaN."""

        return math.isnan(self.get(name))

    @classmethod
    def _handle_color_input(cls, color: ColorInput) -> Self:
        """Handle color input."""

        if isinstance(color, (str, Mapping)):
            return cls.new(color)
        elif cls._is_color(color):
            return color if cls._is_this_color(color) else cls.new(color)  # type: ignore[return-value]
        else:
            raise TypeError(f"Unexpected type '{type(color)}'")

    def space(self) -> str:
        """The current color space."""

        return self._space.NAME

    @classmethod
    def new(
        cls,
        color: ColorInput,
        data: VectorLike | None = None,
        alpha: float = util.DEF_ALPHA,
        **kwargs: Any
    ) -> Self:
        """Create new color object."""

        return cls(color, data, alpha, **kwargs)

    def clone(self) -> Self:
        """Clone."""

        return self.new(self.space(), self[:-1], self[-1])

    def convert(
        self,
        space: str,
        *,
        fit: bool | str = False,
        in_place: bool = False,
        norm: bool = True
    ) -> Self:
        """Convert to color space."""

        # Nothing to do, just return the color with no alterations.
        if space == self.space():
            return self if in_place else self.clone()

        # Actually convert the color
        this = self if in_place else self.clone()
        this._space, this._coords[:-1] = convert.convert(self, space)

        # Normalize achromatic colors, but skip if we internally don't need this.
        if norm and this._space.is_polar() and this.is_achromatic():
            this[this._space.hue_index()] = math.nan  # type: ignore[attr-defined]

        # Fit the color if required
        if fit and not this.in_gamut(tolerance=0.0):
            warn_deprecated(
                "The 'fit' parameter in convert() has been deprecated, please call color.convert(space).fit() instead"
            )
            this.fit(**(fit if isinstance(fit, dict) else {'method': None if fit is True else fit}))

        return this

    def is_achromatic(self) -> bool:
        """Test if color is achromatic."""

        value = self._space.is_achromatic(self.coords(nans=False))
        if value is None:
            xyz = self.convert('xyz-d65')
            return bool(xyz._space.is_achromatic(xyz[:-1]))
        return value

    def mutate(
        self,
        color: ColorInput,
        data: VectorLike | None = None,
        alpha: float = util.DEF_ALPHA,
        **kwargs: Any
    ) -> Self:
        """Mutate the current color to a new color."""

        self._space, self._coords = self._parse(color, data=data, alpha=alpha, **kwargs)
        return self

    def update(
        self,
        color: ColorInput,
        data: VectorLike | None = None,
        alpha: float = util.DEF_ALPHA,
        *,
        norm: bool = True,
        **kwargs: Any
    ) -> Self:
        """Update the existing color space with the provided color."""

        space = self.space()
        self._space, self._coords = self._parse(color, data=data, alpha=alpha, **kwargs)
        if self._space.NAME != space:
            self.convert(space, in_place=True, norm=norm)
        return self

    def _hotswap(self, color: Color) -> Self:
        """
        Hot swap a color object.

        We expect it to be a color object, no special parsing, we just want to go fast.
        """

        self._space, self._coords = self.CS_MAP[color.space()], color[:]
        return self

    def to_string(self, **kwargs: Any) -> str:
        """To string."""

        return self._space.to_string(self, **kwargs)

    def __repr__(self) -> str:
        """Representation."""

        channels = self._space.channels
        l = len(channels)

        return 'color({} {} / {})'.format(
            self._space._serialize()[0],
            ' '.join(
                [
                    util.fmt_float(
                        self[i],
                        util.DEF_PREC,
                        util.DEF_ROUND_MODE
                    ) + POSTFIX[channels[i].angle if channels[i].angle and not math.isnan(self[i]) else ANGLE_NULL]
                    for i in range(l - 1)
                ]
            ),
            util.fmt_float(self[-1], util.DEF_PREC)
        )

    __str__ = __repr__

    def _repr_html_(self) -> str:  # pragma: no cover
        """
        Return an HTML representation of the color for Jupyter and other aware libraries.

        Colors are not gamut mapped, but returned as is.
        """

        svg = ''.join(
            [
                "<svg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 100 100' fill-opacity='0.1'>",
                "<rect width='50' height='50'/>",
                "<rect x='50' y='50' width='50' height='50'/>",
                "</svg>"
            ]
        )

        return ''.join(
            [
                '<div style="margin: 10px 0; text-align: center;">',
                '<div style="box-sizing: border-box; width: 52px; height: 52px; border: 1px solid hsl(0, 0%, 80%);',
                f'background: url(&quot;data:image/svg+xml,{svg}&quot;) 0 0 / 0.75em 0.75em #fefefe; margin: 0 auto;">',
                '<div style="box-sizing: border-box; width: 50px; height: 50px; '
                f'background-color: {self.convert("oklab").to_string()}; border: 2px solid hsl(0, 0%, 90%);"></div>',
                '</div>',
                f'<div>{self.to_string(fit=False)}</div>'
                '</div>'
            ]
        )

    def white(self, cspace: str = 'xyz') -> Vector:
        """Get the white point."""

        value = self.convert_chromaticity('xy-1931', cspace, self._space.WHITE)
        return value if cspace == 'xyz' else value[:-1]

    def uv(self, mode: str = '1976', *, white: VectorLike | None = None) -> Vector:
        """Convert to `xy`."""

        return self.split_chromaticity('uv-' + mode)[:-1]

    def xy(self, *, white: VectorLike | None = None) -> Vector:
        """Convert to `xy`."""

        return self.split_chromaticity('xy-1931')[:-1]

    def split_chromaticity(
        self,
        cspace: str = 'uv-1976',
        *,
        white: VectorLike | None = None
    ) -> Vector:
        """
        Split a color into chromaticity and luminance coordinates.

        Colors are split under the XYZ color space using the current color's white point.
        If results are desired relative to a different white point, one can be provided.
        """

        if white is None:
            white = self._space.WHITE

        # Convert to XYZ D65 as it is a color space that is always required.
        # Chromatically adapt it to the XYZ color space with the current color's white point.
        xyz = self.convert('xyz-d65')
        coords = self.chromatic_adaptation(
            xyz._space.WHITE,
            white,
            xyz.coords(nans=False)
        )

        # XYZ is not a chromaticity space
        if cspace == 'xyz':
            raise ValueError('XYZ is not a luminant-chromaticity color space.')

        # Convert to the requested uv color space if required.
        return (
            self.convert_chromaticity('xyz', cspace, coords, white=white) if cspace != 'xy_1931' else coords
        )

    @classmethod
    def chromaticity(
        cls,
        space: str,
        coords: VectorLike,
        cspace: str = 'uv-1976',
        *,
        scale: bool = False,
        scale_space: str | None = None,
        white: VectorLike | None = None
    ) -> Self:
        """
        Create a color from chromaticity coordinates.

        A luminance of 1 will be assumed unless luminance is included with the coordinates.
        The relative white point of the chromaticity coordinates will be assumed as the
        targeted color space unless one is provided via `white`.

        Lastly, colors can be scaled/normalized within a linear RGB space to normalize
        luminance and provide a nice viewable color. This is useful when the luminance is
        not accurate (such as when luminance is assumed 1). Colors that are out of the linear
        RGB space's gamut will only be rough approximations of the color due to gamut
        limitations. Default linear RGB space is linear sRGB.
        """

        if scale_space is None:
            scale_space = 'srgb-linear'

        # Use the white point of the target color space unless a white point is given.
        if white is None:
            white = cls.CS_MAP[space].WHITE

        # XYZ is not a chromaticity space
        if cspace == 'xyz':
            raise ValueError('XYZ is not a luminant-chromaticity color space.')

        coords = cls.convert_chromaticity(cspace, 'xyz', coords, white=white)

        # Apply chromatic adaptation to match XYZ D65 white point
        color = cls(
            'xyz-d65',
            cls.chromatic_adaptation(white, cls.CS_MAP['xyz-d65'].WHITE, coords)
        )

        # Normalize in the given RGB color space (ideally linear).
        if scale and isinstance(cls.CS_MAP[scale_space], RGBish):
            color.convert(scale_space, in_place=True)
            color[:-1] = util.rgb_scale(color.coords())

        # Convert to targeted color space
        if space != color.space():
            color.convert(space, in_place=True)

        return color

    @classmethod
    def convert_chromaticity(
        cls, cspace1: str,
        cspace2: str,
        coords: VectorLike,
        *,
        white: VectorLike | None = None
    ) -> Vector:
        """
        Convert to or from chromaticity coordinates or between other chromaticity coordinates.

        When converting to or from chromaticity coordinates, the coordinates must be in the XYZ space.
        A white point can be provided and only serves to align colors like black on the achromatic axis;
        otherwise, black will be returned as [0, 0] for the two respective chromaticity points.
        """

        # Check that we know the requested spaces
        if cspace1 not in SUPPORTED_CHROMATICITY_SPACES:
            raise ValueError(f"Unexpected chromaticity space '{cspace1}'")
        if cspace2 not in SUPPORTED_CHROMATICITY_SPACES:
            raise ValueError(f"Unexpected chromaticity space '{cspace2}'")

        # Return if there is nothing to convert
        l = len(coords)
        if (cspace1 == 'xyz' and l != 3) or l not in (2, 3):
            raise ValueError(f'Unexpected number of coordinates ({l}) for {cspace1}')

        # Return if already in desired form
        if cspace1 == cspace2:
            return [*coords, 1] if l == 2 else [*coords]

        # If starting space is XYZ, then convert to xy
        if cspace1 == 'xyz':
            coords = util.xyz_to_xyY(coords, [0.0] * 2 if white is None else white)
            cspace1 = 'xy-1931'

            # If the end space is xy, we have nothing else to do
            if cspace2 == cspace1:
                return coords

        # If we have no luminance, assume 1
        pair, Y = (coords[:-1], coords[-1]) if l == 3 else (coords, 1.0)

        # If we are targeting XYZ, force conversion to xy first.
        target = cspace2
        if cspace2 == 'xyz':
            cspace2 = 'xy-1931'

        # Perform conversion
        if cspace1 == 'xy-1931' and cspace2 != 'xy-1931':
            pair = util.xy_to_uv_1960(pair) if cspace2 == 'uv-1960' else util.xy_to_uv(pair)
        elif cspace1 == 'uv-1960':
            pair = util.uv_1960_to_xy(pair) if cspace2 == 'xy-1931' else util.xy_to_uv(util.uv_1960_to_xy(pair))
        elif cspace1 == 'uv-1976':
            pair = util.uv_to_xy(pair) if cspace2 == 'xy-1931' else util.xy_to_uv_1960(util.uv_to_xy(pair))

        # Special case to convert to XYZ from xy
        if target == 'xyz':
            return util.xy_to_xyz(pair, Y)

        return [*pair, Y]

    @classmethod
    def chromatic_adaptation(
        cls,
        w1: VectorLike,
        w2: VectorLike,
        xyz: VectorLike,
        *,
        method: str | None = None
    ) -> Vector:
        """Chromatic adaptation."""

        adapter = cls.CAT_MAP.get(method if method is not None else cls.CHROMATIC_ADAPTATION)
        if not adapter:
            raise ValueError(f"'{method}' is not a supported CAT")

        return adapter.adapt(tuple(w1), tuple(w2), xyz)  # type: ignore[arg-type]

    def clip(self, space: str | None = None) -> Self:
        """Clip the color channels."""

        orig_space = self.space()
        target_space = space or orig_space

        # We are indirectly clipping this space
        if orig_space != target_space:
            return self.convert(target_space, norm=False, in_place=True).clip().convert(orig_space, in_place=True)

        # Determine what space we actually need to clip in
        if space is None:
            space = self._space.CLIP_SPACE or self._space.GAMUT_CHECK or orig_space
        else:
            cs = self.CS_MAP[space]
            space = cs.CLIP_SPACE or cs.GAMUT_CHECK or cs.NAME

        # Convert to desired space and clip the color
        if space != orig_space:
            conv = self.convert(space, norm=False)
            if not gamut.clip_channels(conv):
                # Clipping only made non-essential changes (normalize hue),
                # just clip in the current space to preserve 'None' and clean up noise
                # at color space boundary limits (if any).
                gamut.clip_channels(self)
                return self
            # Copy results to current color.
            return self._hotswap(conv.convert(orig_space, in_place=True))

        gamut.clip_channels(self)
        return self

    def fit(
        self,
        space: str | None = None,
        *,
        method: str | None = None,
        **kwargs: Any
    ) -> Self:
        """Fit the gamut using the provided method."""

        if method is None:
            method = self.FIT

        # Dedicated clip method.
        if method == 'clip':
            return self.clip(space)

        # If within gamut, just normalize hue range by calling clip.
        if self.in_gamut(space, tolerance=0):
            self.clip(space)
            return self

        # Determine what space we actually need to gamut map in
        if space is None:
            target = self._space.GAMUT_CHECK or self.space()
        else:
            cs = self.CS_MAP[space]
            target = cs.GAMUT_CHECK or cs.NAME

        # Select appropriate mapping algorithm
        mapping = self.FIT_MAP.get(method)
        if not mapping:
            # Unknown fit method
            raise ValueError(f"'{method}' gamut mapping is not currently supported")

        mapping.fit(self, target, **kwargs)
        return self

    def in_gamut(self, space: str | None = None, *, tolerance: float = util.DEF_FIT_TOLERANCE) -> bool:
        """Check if current color is in gamut."""

        if space is None:
            space = self.space()

        # Check if gamut is in the provided space
        c = self.convert(space, norm=False) if space is not None and space != self.space() else self

        # Check the color space specified for gamut checking.
        # If it proves to be in gamut, we will then test if the current
        # space is constrained properly.
        if (
            c._space.GAMUT_CHECK is not None and
            not c.convert(c._space.GAMUT_CHECK, norm=False).in_gamut(tolerance=tolerance)
        ):
            return False

        return gamut.verify(c, tolerance)

    def in_pointer_gamut(self, *, tolerance: float = util.DEF_FIT_TOLERANCE) -> bool:
        """Check if in pointer gamut."""

        return gamut.pointer.in_pointer_gamut(self, tolerance)

    def fit_pointer_gamut(self) -> Self:
        """Check if in pointer gamut."""

        return gamut.pointer.fit_pointer_gamut(self)

    def mask(self, channel: str | Sequence[str], *, invert: bool = False, in_place: bool = False) -> Self:
        """Mask color channels."""

        this = self if in_place else self.clone()
        aliases = self._space.CHANNEL_ALIASES
        masks = set(
            [aliases.get(channel, channel)] if isinstance(channel, str) else [aliases.get(c, c) for c in channel]
        )
        for name in self._space.channels:
            if (not invert and name in masks) or (invert and name not in masks):
                this[name] = math.nan
        return this

    def mix(
        self,
        color: ColorInput,
        percent: float = util.DEF_MIX,
        *,
        in_place: bool = False,
        **interpolate_args: Any
    ) -> Self:
        """
        Mix colors using interpolation.

        This uses the interpolate method to find the center point between the two colors.
        The basic mixing logic is outlined in the CSS level 5 draft.
        """

        # Mix really needs to be between 0 and 1 or steps will break
        domain = interpolate_args.get('domain')
        if domain is not None:
            interpolate_args['domain'] = interpolate.normalize_domain(domain)

        mixed = self.interpolate([self, color], **interpolate_args)(percent)  # type: Self
        return self._hotswap(mixed) if in_place else mixed

    @classmethod
    def steps(
        cls,
        colors: Sequence[ColorInput | interpolate.stop | Callable[..., float]],
        *,
        steps: int = 2,
        max_steps: int = 1000,
        max_delta_e: float = 0,
        delta_e: str | None = None,
        delta_e_args: dict[str, Any] | None = None,
        **interpolate_args: Any
    ) -> list[Self]:
        """Discrete steps."""

        # Scale really needs to be between 0 and 1 or steps will break
        domain = interpolate_args.get('domain')
        if domain is not None:
            interpolate_args['domain'] = interpolate.normalize_domain(domain)

        return cls.interpolate(colors, **interpolate_args).steps(steps, max_steps, max_delta_e, delta_e, delta_e_args)

    @classmethod
    def discrete(
        cls,
        colors: Sequence[ColorInput | interpolate.stop | Callable[..., float]],
        *,
        space: str | None = None,
        out_space: str | None = None,
        steps: int | None = None,
        max_steps: int = 1000,
        max_delta_e: float = 0,
        delta_e: str | None = None,
        delta_e_args: dict[str, Any] | None = None,
        domain: Vector | None = None,
        **interpolate_args: Any
    ) -> Interpolator[Self]:
        """Create a discrete interpolation."""

        # If no steps were provided, use the number of colors provided
        num = sum((not callable(c) or not isinstance(c, interpolate.stop)) for c in colors) if steps is None else steps
        i = cls.interpolate(colors, space=space, **interpolate_args)
        # Convert the interpolation into a discretized interpolation with the requested number of steps
        i = i.discretize(num, max_steps, max_delta_e, delta_e, delta_e_args)
        if domain is not None:
            i.domain(domain)
        if out_space is not None:
            i.out_space(out_space)
        return i

    @classmethod
    def interpolate(
        cls,
        colors: Sequence[ColorInput | interpolate.stop | Callable[..., float]],
        *,
        space: str | None = None,
        out_space: str | None = None,
        progress: Mapping[str, Callable[..., float]] | Callable[..., float] | None = None,
        hue: str = util.DEF_HUE_ADJ,
        premultiplied: bool = True,
        extrapolate: bool = False,
        domain: Vector | None = None,
        method: str | None = None,
        padding: float | tuple[float, float] | None = None,
        carryforward: bool | None = None,
        powerless: bool | None = None,
        **kwargs: Any
    ) -> Interpolator[Self]:
        """
        Return an interpolation function.

        The function will return an interpolation function that accepts a value (which should
        be in the range of [0..1] and will return a color based on that value.

        While we use NaNs to mask off channels when doing the interpolation, we do not allow
        arbitrary specification of NaNs by the user, they must specify channels via `adjust`
        if they which to target specific channels for mixing. Null hues become NaNs before
        mixing occurs.
        """

        return interpolate.interpolator(
            cls,
            method if method is not None else cls.INTERPOLATOR,
            colors=colors,
            space=space,
            out_space=out_space,
            progress=progress,
            hue=hue,
            premultiplied=premultiplied,
            extrapolate=extrapolate,
            domain=domain,
            padding=padding,
            carryforward=carryforward if carryforward is not None else cls.CARRYFORWARD,
            powerless=powerless if powerless is not None else cls.POWERLESS,
            **kwargs
        )

    @classmethod
    def average(
        cls,
        colors: Iterable[ColorInput],
        weights: Iterable[float] | None = None,
        *,
        space: str | None = None,
        out_space: str | None = None,
        premultiplied: bool = True,
        **kwargs: Any
    ) -> Self:
        """Average the colors."""

        if space is None:
            space = cls.AVERAGE

        if out_space is None:
            out_space = space

        return average.average(
            cls,
            colors,
            weights,
            space,
            premultiplied
        ).convert(out_space, in_place=True)

    def filter(  # noqa: A003
        self,
        name: str,
        amount: float | None = None,
        *,
        space: str | None = None,
        out_space: str | None = None,
        in_place: bool = False,
        **kwargs: Any
    ) -> Self:
        """Filter."""

        return filters.filters(self, name, amount, space, out_space, in_place, **kwargs)

    def harmony(
        self,
        name: str,
        *,
        space: str | None = None,
        out_space: str | None = None,
        **kwargs: Any
    ) -> list[Self]:
        """Acquire the specified color harmonies."""

        if space is None:
            space = self.HARMONY

        if out_space is None:
            out_space = space

        return [c.convert(out_space, in_place=True) for c in harmonies.harmonize(self, name, space, **kwargs)]

    @classmethod
    def layer(
        cls,
        colors: Sequence[ColorInput],
        *,
        blend: str | bool = 'normal',
        operator: str | bool = 'source-over',
        space: str | None = None,
        out_space: str | None = None
    ) -> Self:
        """
        Apply color compositing (blend modes and alpha blending) on a list of colors.

        Colors are overlaid on each other with left being the top of the stack and right being the bottom of the stack.
        """

        if isinstance(blend, bool):
            b = 'normal' if blend else None
            warn_deprecated("The use of boolean values for 'blend' has been deprecated, please use a specific mode")
        else:
            b = blend

        if isinstance(operator, bool):
            o = 'source-over' if operator else None
            warn_deprecated("The use of boolean values for 'blend' has been deprecated, please use a specific mode")
        else:
            o = operator

        return compositing.compose(cls, colors, b, o, space, out_space)

    def delta_e(
        self,
        color: ColorInput,
        *,
        method: str | None = None,
        **kwargs: Any
    ) -> float:
        """Delta E distance."""

        color = self._handle_color_input(color)
        if method is None:
            method = self.DELTA_E

        delta = self.DE_MAP.get(method)
        if not delta:
            raise ValueError(f"'{method}' is not currently a supported distancing algorithm.")
        return delta.distance(self, color, **kwargs)

    def distance(self, color: ColorInput, *, space: str = "lab") -> float:
        """Delta."""

        return distance.distance_euclidean(self, self._handle_color_input(color), space=space)

    def closest(
        self,
        colors: Sequence[ColorInput],
        *,
        method: str | None = None,
        **kwargs: Any
    ) -> Self:
        """Find the closest color to the current base color."""

        return distance.closest(self, colors, method=method, **kwargs)

    def luminance(self, *, white: VectorLike | None = cat.WHITES['2deg']['D65']) -> float:
        """Get color's luminance."""

        if white is None:
            white = self._space.WHITE

        # Convert to XYZ D65 as it is a color space that is always required.
        # Chromatically adapt it to the XYZ color space with the current color's white point.
        xyz = self.convert('xyz-d65')
        coords = self.chromatic_adaptation(
            xyz._space.WHITE,
            white,
            xyz.coords(nans=False)
        )

        return coords[1]

    def contrast(self, color: ColorInput, method: str | None = None) -> float:
        """Compare the contrast ratio of this color and the provided color."""

        color = self._handle_color_input(color)
        return contrast.contrast(method, self, color)

    @overload
    def get(self,
        name: str,
        *,
        nans: bool = ...,
        precision: int | Sequence[int] | None = ...,
        rounding: str | None = ...
    ) -> float:
        ...

    @overload
    def get(
        self,
        name: list[str] | tuple[str, ...],
        *,
        nans: bool = ...,
        precision: int | Sequence[int] | None = ...,
        rounding: str | None = ...
    ) -> Vector:
        ...

    def get(
        self,
        name: str | list[str] | tuple[str, ...],
        *,
        nans: bool = True,
        precision: int | Sequence[int] | None = None,
        rounding: str | None = None
    ) -> float | Vector:
        """Get channel."""

        if rounding is None:
            rounding = self.ROUNDING
        is_plist = precision is not None and not isinstance(precision, int)

        # Handle single channel
        if isinstance(name, str):
            # Handle space.channel
            if '.' in name:
                space, channel = name.split('.', 1)
                obj = self.convert(space, norm=nans)
                if nans:
                    v = obj[channel]
                else:
                    i = obj._space.get_channel_index(channel)
                    v = obj._space.resolve_channel(i, obj._coords)
            elif nans:
                v = self[name]
            else:
                i = self._space.get_channel_index(name)
                v = self._space.resolve_channel(i, self._coords)

            if precision is None:
                return v

            return alg.round_to(
                v,
                util.get_index(precision, 0) if is_plist else precision,  # type: ignore[arg-type]
                rounding
            )

        # Handle list of channels
        else:
            original_space = current_space = self.space()
            obj = self
            values = []

            for e, n in enumerate(name):
                # Handle space.channel
                space, channel = n.split('.', 1) if '.' in n else (original_space, n)
                if space != current_space:
                    obj = self if space == original_space else self.convert(space, norm=nans)
                    current_space = space
                if nans:
                    v = obj[channel]
                else:
                    i = obj._space.get_channel_index(channel)
                    v = obj._space.resolve_channel(i, obj._coords)

                if precision is None:
                    values.append(v)
                    continue

                values.append(
                    alg.round_to(
                        v,
                        util.get_index(precision, e) if is_plist else precision,  # type: ignore[arg-type]
                        rounding
                    )
                )
            return values

    def set(  # noqa: A003
        self,
        name: str | dict[str, float | Callable[..., float]],
        value: float | Callable[..., float] | None = None,
        *,
        nans: bool = True
    ) -> Self:
        """Set channel."""

        # Set all the channels in a dictionary.
        # Sort by name to reduce how many times we convert
        # when dealing with different color spaces.
        if value is None:
            if isinstance(name, str):
                raise ValueError(f"Missing the positional 'value' argument for channel '{name}'")

            original_space = current_space = self.space()
            obj = self.clone()

            for k, v in name.items():
                # Handle space.channel
                space, channel = k.split('.', 1) if '.' in k else (original_space, k)
                if space != current_space:
                    obj.convert(space, in_place=True, norm=nans)
                    current_space = space
                if not callable(v):
                    obj[channel] = v
                else:
                    i = obj._space.get_channel_index(channel)
                    obj[channel] = v(obj[i] if nans else obj._space.resolve_channel(i, obj._coords))

            # Update the original color
            self.update(obj)

        # Set a single channel value
        else:
            if isinstance(name, dict):
                raise ValueError("A dict of channels and values cannot be used with the positional 'value' parameter")

            # Handle space.channel
            if '.' in name:
                space, channel = name.split('.', 1)
                obj = self.convert(space, norm=nans)
                if not callable(value):
                    obj[channel] = value
                else:
                    i = obj._space.get_channel_index(channel)
                    obj[channel] = value(obj[i] if nans else obj._space.resolve_channel(i, obj._coords))
                return self.update(obj)

            # Handle a function that modifies the value or a direct value
            if not callable(value):
                self[name] = value
            else:
                i = self._space.get_channel_index(name)
                self[name] = value(self[i] if nans else self._space.resolve_channel(i, self._coords))

        return self

    def coords(
        self,
        *,
        nans: bool = True,
        precision: int | Sequence[int] | None = None,
        rounding: str | None = None
    ) -> Vector:
        """Get the color channels and optionally remove undefined values."""

        # Full precision
        if precision is None:
            if nans:
                return self[:-1]
            else:
                return [
                    self._space.resolve_channel(index, self._coords)
                    for index in range(len(self._coords) - 1)
                ]

        pint = isinstance(precision, int)
        if rounding is None:
            rounding = self.ROUNDING

        return [
            alg.round_to(
                self[index] if nans else self._space.resolve_channel(index, self._coords),
                precision if pint else util.get_index(precision, index, self.PRECISION),  # type: ignore[arg-type]
                rounding
            )
            for index in range(len(self._coords) - 1)
        ]

    def alpha(
        self,
        *,
        nans: bool = True,
        precision: int | None = None,
        rounding: str | None = None
    ) -> float:
        """Get the alpha channel."""

        value = self[-1] if nans else self._space.resolve_channel(-1, self._coords)
        if precision is None:
            return value
        return alg.round_to(value, precision, self.ROUNDING if rounding is None else rounding) if precision else value


Color.register(
    [
        # Spaces
        XYZD65(),
        XYZD50(),
        sRGB(),
        sRGBLinear(),
        DisplayP3(),
        DisplayP3Linear(),
        Oklab(),
        OkLCh(),
        Lab(),
        LCh(),
        LabD65(),
        LChD65(),
        Jzazbz(),
        JzCzhz(),
        ICtCp(),
        HSV(),
        HSL(),
        HWB(),
        Rec2020(),
        Rec2020Linear(),
        Rec2100PQ(),
        Rec2100HLG(),
        Rec2100Linear(),
        A98RGB(),
        A98RGBLinear(),
        ProPhotoRGB(),
        ProPhotoRGBLinear(),

        # CAT
        Bradford(),

        # Delta E
        DE76(),
        DE94(),
        DECMC(),
        DE2000(),
        DEHyAB(),
        DEOK(),
        DEITP(),
        DEZ(),

        # Fit
        MINDEChroma(),
        LChChroma(),
        OkLChChroma(),
        RayTrace(),

        # Filters
        Sepia(),
        Brightness(),
        Contrast(),
        Saturate(),
        Opacity(),
        HueRotate(),
        Grayscale(),
        Invert(),
        Protan(),
        Deutan(),
        Tritan(),

        # Contrast
        WCAG21Contrast(),

        # Interpolation
        Linear(),
        CSSLinear(),
        Continuous(),
        BSpline(),
        NaturalBSpline(),
        Monotone(),

        # CCT
        Robertson1968(),
        Ohno2013()
    ]
)
