import sublime
import os
from pathlib import Path
from .constants import SETTINGS_FILE

def claudette_chat_status_message(window, message: str, prefix: str = "ℹ️") -> int:
    """
    Display a status message in the active chat view.

    Args:
        window: The Sublime Text window
        message (str): The status message to display
        prefix (str, optional): Icon or text prefix for the message. Defaults to "ℹ️"

    Returns:
        int: The end position of the message in the view, or -1 if no view was found
    """
    if not window:
        return -1

    # Find the active chat view
    current_chat_view = None
    for view in window.views():
        if (view.settings().get('claudette_is_chat_view', False) and
            view.settings().get('claudette_is_current_chat', False)):
            current_chat_view = view
            break

    if not current_chat_view:
        return -1

    if current_chat_view.size() > 0:
        view_size = current_chat_view.size()
        last_chars = current_chat_view.substr(sublime.Region(max(0, view_size - 2), view_size))
        if last_chars == '\n\n':
            # Content already ends with two newlines, don't add any newline before message
            message = f"{prefix + ' ' if prefix else ''}{message}\n"
        elif last_chars.endswith('\n'):
            # Content ends with one newline, add one more for spacing
            message = f"\n{prefix + ' ' if prefix else ''}{message}\n"
        else:
            # Content doesn't end with newline, add two for spacing
            message = f"\n\n{prefix + ' ' if prefix else ''}{message}\n"
    else:
        message = f"{prefix + ' ' if prefix else ''}{message}\n"

    current_chat_view.set_read_only(False)
    current_chat_view.run_command('append', {
        'characters': message,
        'force': True,
        'scroll_to_end': True
    })

    end_point = current_chat_view.size()

    current_chat_view.sel().clear()
    current_chat_view.sel().add(sublime.Region(end_point, end_point))

    current_chat_view.set_read_only(True)

    return end_point

def claudette_estimate_api_tokens(text):
    """Estimate Claude API tokens based on character count (rough approximation)."""
    return len(text) // 4

def claudette_detect_encoding(sample):
    """
    Detect file encoding using BOMs and content analysis.
    Similar to how Sublime Text handles encodings.
    """
    # Check for BOMs
    if sample.startswith(b'\xEF\xBB\xBF'):
        return 'utf-8-sig'
    elif sample.startswith(b'\xFE\xFF'):
        return 'utf-16be'
    elif sample.startswith(b'\xFF\xFE'):
        return 'utf-16le'
    elif sample.startswith(b'\x00\x00\xFE\xFF'):
        return 'utf-32be'
    elif sample.startswith(b'\xFF\xFE\x00\x00'):
        return 'utf-32le'

    # Try UTF-8
    try:
        sample.decode('utf-8')
        return 'utf-8'
    except UnicodeDecodeError:
        return 'latin-1'  # Fallback encoding

def claudette_is_text_file(file_path, sample_size=4096, max_size=1024*1024*10):
    """
    More complete implementation of Sublime Text's text file detection.

    Args:
        file_path: Path to the file to check
        sample_size: Number of bytes to sample
        max_size: Maximum file size to consider (10MB default)

    Returns:
        tuple: (is_text, encoding, reason)
    """
    try:
        file_size = os.path.getsize(file_path)

        # Size check
        if file_size > max_size:
            return False, None, "File too large"

        # Empty file check
        if file_size == 0:
            return True, 'utf-8', "Empty file"

        with open(file_path, 'rb') as f:
            sample = f.read(min(sample_size, file_size))

        # Binary check
        if b'\x00' in sample:
            null_percentage = sample.count(b'\x00') / len(sample)
            if null_percentage > 0.01:  # More than 1% nulls
                return False, None, "Binary file (contains NULL bytes)"

        # Encoding detection
        encoding = claudette_detect_encoding(sample)

        # Verification check
        try:
            with open(file_path, 'r', encoding=encoding) as f:
                f.read(sample_size)
            return True, encoding, "Valid text file"
        except UnicodeDecodeError:
            return False, None, "Unable to decode with detected encoding"

    except IOError as e:
        return False, None, f"IO Error: {str(e)}"

def claudette_get_api_key():
    """
    Get the currently active API key.

    Returns:
        dict or None: The active API key as a dictionary with 'key' and 'name' fields,
                      or None if no valid key is available
    """
    settings = sublime.load_settings(SETTINGS_FILE)
    api_key = settings.get('api_key')

    # For string API key, return a dict format
    if isinstance(api_key, str) and api_key.strip():
        return {'key': api_key, 'name': 'Default'}

    # For dict with multiple keys, get the current one
    elif isinstance(api_key, dict) and api_key.get('keys') and isinstance(api_key['keys'], list):
        keys = api_key['keys']
        current_index = api_key.get('active_key', 0)

        # If there's a valid current index, return that key
        if isinstance(current_index, int) and 0 <= current_index < len(keys):
            key_entry = keys[current_index]
            if isinstance(key_entry, dict) and key_entry.get('key'):
                return key_entry

        # Otherwise return the first valid key
        for key_entry in keys:
            if isinstance(key_entry, dict) and key_entry.get('key'):
                return key_entry

    return None

def claudette_get_api_key_value():
    """
    Extract the API key value from the API key dictionary.

    Returns:
        str: The API key value, or an empty string if not available
    """
    api_key = claudette_get_api_key()

    if isinstance(api_key, dict):
        return api_key.get('key', '')

    return ''

def claudette_get_api_key_name():
    """
    Get the API key name from the API key dictionary.

    Returns:
        str: The API key name, 'Default' if the key has no name, or 'Undefined'
    """
    api_key = claudette_get_api_key()

    if isinstance(api_key, dict):
        return api_key.get('name', 'Default')

    return 'Undefined'
