import sublime
import sublime_plugin
import subprocess
import tempfile
import os
import json
import threading
import sys
import shutil


class AiCompleteCommand(sublime_plugin.TextCommand):
    def run(self, edit):
        settings    = sublime.load_settings("AiComplete.sublime-settings")
        api_key     = settings.get("gemini_api_key", "")
        default_coding_prompt = settings.get("default_coding_prompt", "")
        user_prompt           = settings.get("user_prompt", "")

        # Get file extension of the current file
        file_name = self.view.file_name()
        file_ext = os.path.splitext(file_name)[1][1:] if file_name else ""

        file_ext_prompt = (
            ".{0} is the extension of this file.".format(file_ext)
            if file_ext else ""
        )

        if not api_key:
            sublime.error_message("Gemini API key not set. Run 'AIComplete : Open Settings' from Command Palette (Cmd/Ctrl + Shift + P) to add it.")
            return

        for region in self.view.sel():
            if region.empty():
                line_region = self.view.line(region)
                user_input = self.view.substr(line_region).strip()
                insert_point = line_region.end()
            else:
                user_input = self.view.substr(region).strip()
                insert_point = region.end()

            if not user_input:
                sublime.status_message("AIComplete : No text selected or on line.")
                return

            prompt_text = (
                "You are an assistant which outputs code and only code in plain text. "
                "Output ONLY code, with NO markdown formatting whatsoever. Do NOT use backticks (`). "
                "Ensure the output is pure, unformatted text. No code blocks, no language identifiers. "
                "The output must be identical to what you would copy directly into a plain text editor, "
                "without any added characters. Do not add comments. Please act according to user instructions. "
                "User Instructions: " + user_input + " "
                "User Additional Instructions: " + default_coding_prompt + " " + file_ext_prompt + " " + user_prompt
            )

            self.view.show_popup(
                " Getting a response from AI ",
                flags=sublime.HIDE_ON_MOUSE_MOVE_AWAY,
                location=insert_point,
                max_width=440
            )

            threading.Thread(
                target=self.get_and_insert_response,
                args=(prompt_text, insert_point, api_key),
                daemon=True
            ).start()

    def get_and_insert_response(self, prompt_text, insert_point, api_key):
        gemini_response = call_gemini_via_subprocess(api_key, prompt_text)
        cleaned_response = '\n'.join(
            line for line in gemini_response.splitlines() if '```' not in line
        )

        sublime.set_timeout(lambda: self.insert_and_close_popup(insert_point, cleaned_response), 0)

    def insert_and_close_popup(self, insert_point, cleaned_response):
        self.view.hide_popup()
        self.view.run_command("insert_text_at_point", {
            "point": insert_point,
            "text": "\n" + cleaned_response + "\n"
        })

class InsertTextAtPointCommand(sublime_plugin.TextCommand):
    def run(self, edit, point, text):
        self.view.insert(edit, point, text)

class OpenGeminiSettingsCommand(sublime_plugin.ApplicationCommand):
    def run(self):
        sublime.run_command("edit_settings", {
            "base_file": "${packages}/User/AiComplete.sublime-settings",
            "default": "{\n  \"gemini_api_key\": \"your-api-key-here\"\n}"
        })

def plugin_loaded():
    user_settings_path = os.path.join(sublime.packages_path(), "User", "AiComplete.sublime-settings")

    if not os.path.exists(user_settings_path):
        try:
            contents = sublime.load_resource("Packages/AiComplete/AiComplete.sublime-settings")
            with open(user_settings_path, "w", encoding="utf-8") as f:
                f.write(contents)
            print("[AIComplete] Default settings copied to User folder.")
        except Exception as e:
            print("[AIComplete] Failed to copy default settings:", e)

    threading.Thread(target=check_google_sdk_installed, daemon=True).start()

def check_google_sdk_installed():
    try:
        python_path = shutil.which("python3") or "python3"
        print(python_path)
        test_code = "import google.generativeai; print('OK')"

        output = subprocess.check_output(
            [python_path, "-c", test_code],
            stderr=subprocess.STDOUT,
            universal_newlines=True
        )

        if "OK" in output:
            sublime.status_message("[AIComplete] google-generativeai is available.")
        else:
            raise Exception("Not found")

    except Exception as e:
        msg = (
            "[AIComplete] Missing required dependency: google-generativeai\n\n"
            "To install it, run the following command in your terminal:\n\n"
            "{} -m pip install google-generativeai"
        ).format(python_path)
        print(msg)
        sublime.set_timeout(lambda: sublime.error_message(msg), 0)

def call_gemini_via_subprocess(api_key, prompt_text):
    try:
        with tempfile.NamedTemporaryFile(mode="w+", suffix=".py", delete=False) as temp:
            script_path = temp.name
            temp.write(GEMINI_SDK_SCRIPT.format(
                api_key=json.dumps(api_key),
                prompt=json.dumps(prompt_text)
            ))

        output = subprocess.check_output(
            ["python3", script_path],
            stderr=subprocess.STDOUT,
            timeout=30
        )
        return output.decode("utf-8").strip()

    except subprocess.CalledProcessError as e:
        return "[Subprocess error: {}]".format(e.output.decode("utf-8", errors="ignore"))
    except Exception as e:
        return "[Exception: {}]".format(e)
    finally:
        if os.path.exists(script_path):
            os.remove(script_path)

GEMINI_SDK_SCRIPT = '''\
import google.generativeai as genai
import json
import sys

api_key = {api_key}
prompt = {prompt}

genai.configure(api_key=api_key)

model = genai.GenerativeModel("gemini-2.0-flash")

try:
    response = model.generate_content(prompt)
    print(response.text)
except Exception as e:
    print("[Gemini SDK Error]", e)
'''
